<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\GraphQLCompose\FieldType;

use Drupal\Core\Field\FieldItemInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerItemInterface;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerTrait;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeFieldTypeBase;
use Drupal\twenty_crm\Helper\PersonHelper;
use Drupal\twenty_crm\Plugin\Field\FieldType\CustomerReferenceItem;
use Drupal\twenty_crm\TwentyCrmPersonService;

/**
 * {@inheritDoc}
 *
 * @GraphQLComposeFieldType(
 *   id = "customer_reference",
 *   type_sdl = "TwentyCrmCustomerReference",
 * )
 */
class CustomerReferenceFieldType extends GraphQLComposeFieldTypeBase implements FieldProducerItemInterface {

  use FieldProducerTrait;

  /**
   * {@inheritdoc}
   */
  public function resolveFieldItem(FieldItemInterface $item, FieldContext $context) {
    if ($item instanceof CustomerReferenceItem) {
      $data = [
        'contactUuid' => $item->contact_uuid,
        'companyUuid' => $item->company_uuid,
        'companyName' => $item->company_name,
        'contactName' => $item->contact_name,
      ];

      // If we have a contact UUID, fetch additional contact data from the CRM.
      if (!empty($item->contact_uuid)) {
        $data['contact'] = $this->getPersonData($item->contact_uuid);
      }

      // If we have a company UUID, reuse the company data method from CompanyReferenceFieldType.
      if (!empty($item->company_uuid)) {
        $data['company'] = CompanyReferenceFieldType::getCompanyData($item->company_uuid);
      }

      return $data;
    }
  }

  /**
   * Get person data from Twenty CRM API.
   *
   * @param string $personUuid
   *   The person UUID.
   *
   * @return array|null
   *   The person data or NULL if not found.
   */
  private function getPersonData(string $personUuid): ?array {
    /** @var \Drupal\twenty_crm\TwentyCrmPersonService $personService */
    $personService = \Drupal::service('twenty_crm.person_service');
    $person = $personService->getPersonById($personUuid);

    if (!$person) {
      return NULL;
    }

    return [
      'id' => $person->getId(),
      'email' => PersonHelper::getPrimaryEmail($person),
      'firstName' => PersonHelper::getFirstName($person),
      'lastName' => PersonHelper::getLastName($person),
      'fullName' => PersonHelper::getFullName($person),
      'jobTitle' => PersonHelper::getJobTitle($person),
      'companyId' => PersonHelper::getCompanyId($person),
      'createdAt' => $person->get('createdAt'),
      'updatedAt' => $person->get('updatedAt'),
    ];
  }

}
