<?php

declare(strict_types=1);

namespace Drupal\twenty_crm;

use Drupal\Core\Logger\LoggerChannelInterface;
use Factorial\TwentyCrm\Client\ClientInterface;
use Factorial\TwentyCrm\Collection\DynamicEntityCollection;
use Factorial\TwentyCrm\DTO\SearchOptions;
use Factorial\TwentyCrm\Entity\DynamicEntity;
use Factorial\TwentyCrm\Query\FilterInterface;

/**
 * Drupal wrapper service for Twenty CRM company operations.
 */
final class TwentyCrmCompanyService {

  /**
   * Constructs a TwentyCrmCompanyService object.
   *
   * @param \Factorial\TwentyCrm\Client\ClientInterface $client
   *   The Twenty CRM client.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(
    protected readonly ClientInterface $client,
    protected readonly LoggerChannelInterface $logger,
  ) {}

  /**
   * Find companies using the Twenty CRM API.
   *
   * @param \Factorial\TwentyCrm\Query\FilterInterface $filter
   *   The search filter.
   * @param \Factorial\TwentyCrm\DTO\SearchOptions $options
   *   The search options.
   *
   * @return \Factorial\TwentyCrm\Collection\DynamicEntityCollection
   *   The collection with company results.
   */
  public function findCompanies(FilterInterface $filter, SearchOptions $options): DynamicEntityCollection {
    try {
      $this->logger->info('Twenty CRM: Searching companies', [
        'limit' => $options->limit,
      ]);

      return $this->client->entity('company')->find($filter, $options);
    }
    catch (\Exception $e) {
      $this->logger->error('Company search failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      $definition = $this->client->entity('company')->getDefinition();
      return new DynamicEntityCollection($definition, []);
    }
  }

  /**
   * Get a company by UUID.
   *
   * @param string $id
   *   The company UUID.
   *
   * @return \Factorial\TwentyCrm\Entity\DynamicEntity|null
   *   The company entity or NULL if not found/error.
   */
  public function getCompanyById(string $id): ?DynamicEntity {
    try {
      return $this->client->entity('company')->getById($id);
    }
    catch (\Exception $e) {
      $this->logger->error('Company fetch by ID failed for @id: @message', [
        '@id' => $id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
