<?php

declare(strict_types=1);

namespace Drupal\twenty_crm;

use Drupal\Core\Logger\LoggerChannelInterface;
use Factorial\TwentyCrm\Client\ClientInterface;
use Factorial\TwentyCrm\Collection\DynamicEntityCollection;
use Factorial\TwentyCrm\DTO\SearchOptions;
use Factorial\TwentyCrm\Entity\DynamicEntity;
use Factorial\TwentyCrm\Query\FilterInterface;

/**
 * Drupal wrapper for Twenty CRM Person service.
 */
final class TwentyCrmPersonService {

  /**
   * Constructs a TwentyCrmPersonService object.
   */
  public function __construct(
    protected readonly ClientInterface $client,
    protected readonly LoggerChannelInterface $logger,
  ) {}

  /**
   * Find people based on search criteria.
   *
   * @param \Factorial\TwentyCrm\Query\FilterInterface $filter
   *   The search filter.
   * @param \Factorial\TwentyCrm\DTO\SearchOptions $options
   *   The search options (pagination, sorting, etc.).
   *
   * @return \Factorial\TwentyCrm\Collection\DynamicEntityCollection
   *   The collection with person results.
   */
  public function findPeople(FilterInterface $filter, SearchOptions $options): DynamicEntityCollection {
    try {
      $this->logger->info('Twenty CRM: Searching people', [
        'limit' => $options->limit,
      ]);

      return $this->client->entity('person')->find($filter, $options);

    }
    catch (\Exception $e) {
      $this->logger->error('Twenty CRM person search failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      $definition = $this->client->entity('person')->getDefinition();
      return new DynamicEntityCollection($definition, []);
    }
  }

  /**
   * Get a person by ID.
   *
   * @param string $person_id
   *   The person ID.
   *
   * @return \Factorial\TwentyCrm\Entity\DynamicEntity|null
   *   The person entity or NULL if not found.
   */
  public function getPersonById(string $person_id): ?DynamicEntity {
    try {
      return $this->client->entity('person')->getById($person_id);
    }
    catch (\Exception $e) {
      $this->logger->error('Twenty CRM get person by ID failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
