<?php

declare(strict_types=1);

namespace Drupal\Tests\twenty_crm\ExistingSite;

use Symfony\Component\HttpFoundation\Response;
use weitzman\DrupalTestTraits\ExistingSiteBase;

/**
 * Tests Twenty CRM autocomplete endpoints.
 *
 * @group twenty_crm
 */
final class AutocompleteControllerTest extends ExistingSiteBase {

  /**
   * Test company autocomplete returns valid results.
   */
  public function testCompanyAutocompleteReturnsResults(): void {
    $this->drupalGet('/twenty-crm/autocomplete/company', ['query' => ['q' => 'factorial']]);
    $this->assertSession()->statusCodeEquals(Response::HTTP_OK);

    $content = $this->getSession()->getPage()->getContent();
    $results = json_decode($content, TRUE);

    self::assertIsArray($results, 'Response should be a JSON array');
    self::assertNotEmpty($results, 'Should return at least one company');

    // Verify structure of first result.
    $first = $results[0];
    self::assertArrayHasKey('uuid', $first, 'Result should have uuid field');
    self::assertArrayHasKey('name', $first, 'Result should have name field');
    self::assertArrayHasKey('location', $first, 'Result should have location field');
    self::assertArrayHasKey('city', $first, 'Result should have city field');
    self::assertArrayHasKey('country', $first, 'Result should have country field');

    // Verify uuid is not null (was a bug with direct property access).
    self::assertNotNull($first['uuid'], 'UUID should not be null');
    self::assertNotEmpty($first['uuid'], 'UUID should not be empty');

    // Verify name contains search term.
    self::assertStringContainsStringIgnoringCase('factorial', $first['name'], 'Name should contain search term');
  }

  /**
   * Test company autocomplete with short query returns empty.
   */
  public function testCompanyAutocompleteShortQueryReturnsEmpty(): void {
    $this->drupalGet('/twenty-crm/autocomplete/company', ['query' => ['q' => 'f']]);
    $this->assertSession()->statusCodeEquals(Response::HTTP_OK);

    $content = $this->getSession()->getPage()->getContent();
    $results = json_decode($content, TRUE);

    self::assertIsArray($results, 'Response should be a JSON array');
    self::assertEmpty($results, 'Should return empty array for single character query');
  }

  /**
   * Test company autocomplete with no match returns empty.
   */
  public function testCompanyAutocompleteNoMatchReturnsEmpty(): void {
    $this->drupalGet('/twenty-crm/autocomplete/company', ['query' => ['q' => 'xyznonexistent12345']]);
    $this->assertSession()->statusCodeEquals(Response::HTTP_OK);

    $content = $this->getSession()->getPage()->getContent();
    $results = json_decode($content, TRUE);

    self::assertIsArray($results, 'Response should be a JSON array');
    self::assertEmpty($results, 'Should return empty array for non-matching query');
  }

  /**
   * Test person autocomplete with valid company UUID.
   */
  public function testPersonAutocompleteWithCompanyUuid(): void {
    // First get a company UUID.
    $this->drupalGet('/twenty-crm/autocomplete/company', ['query' => ['q' => 'factorial']]);
    $content = $this->getSession()->getPage()->getContent();
    $companies = json_decode($content, TRUE);

    self::assertNotEmpty($companies, 'Need at least one company for person test');
    $companyUuid = $companies[0]['uuid'];

    // Now search for people in that company.
    // Note: This may return empty if no people are associated with the company.
    $this->drupalGet("/twenty-crm/autocomplete/person/{$companyUuid}", ['query' => ['q' => 'st']]);
    $this->assertSession()->statusCodeEquals(Response::HTTP_OK);

    $content = $this->getSession()->getPage()->getContent();
    $results = json_decode($content, TRUE);

    self::assertIsArray($results, 'Response should be a JSON array');

    // If results exist, verify structure.
    if (!empty($results)) {
      $first = $results[0];
      self::assertArrayHasKey('uuid', $first, 'Result should have uuid field');
      self::assertArrayHasKey('name', $first, 'Result should have name field');
      self::assertArrayHasKey('email', $first, 'Result should have email field');
      self::assertArrayHasKey('jobTitle', $first, 'Result should have jobTitle field');

      // Verify uuid is not null.
      self::assertNotNull($first['uuid'], 'UUID should not be null');
    }
  }

  /**
   * Test person autocomplete with missing company UUID returns empty.
   */
  public function testPersonAutocompleteWithoutCompanyReturnsEmpty(): void {
    $this->drupalGet('/twenty-crm/autocomplete/person/__none__', ['query' => ['q' => 'test']]);
    $this->assertSession()->statusCodeEquals(Response::HTTP_OK);

    $content = $this->getSession()->getPage()->getContent();
    $results = json_decode($content, TRUE);

    self::assertIsArray($results, 'Response should be a JSON array');
    self::assertEmpty($results, 'Should return empty array when company is __none__');
  }

  /**
   * Test person autocomplete with short query returns empty.
   */
  public function testPersonAutocompleteShortQueryReturnsEmpty(): void {
    // First get a company UUID.
    $this->drupalGet('/twenty-crm/autocomplete/company', ['query' => ['q' => 'factorial']]);
    $content = $this->getSession()->getPage()->getContent();
    $companies = json_decode($content, TRUE);

    if (empty($companies)) {
      $this->markTestSkipped('No companies available to test person autocomplete');
    }

    $companyUuid = $companies[0]['uuid'];

    // Single character query should return empty.
    $this->drupalGet("/twenty-crm/autocomplete/person/{$companyUuid}", ['query' => ['q' => 's']]);
    $this->assertSession()->statusCodeEquals(Response::HTTP_OK);

    $content = $this->getSession()->getPage()->getContent();
    $results = json_decode($content, TRUE);

    self::assertIsArray($results, 'Response should be a JSON array');
    self::assertEmpty($results, 'Should return empty array for single character query');
  }

}
