# Twenty CRM Integration

A comprehensive Drupal module that provides seamless integration with the Twenty CRM API for customer relationship management, featuring an advanced Tagify-based UI component for customer reference fields.

## Table of Contents

- [Features](#features)
- [Requirements](#requirements)
- [Installation](#installation)
- [Configuration](#configuration)
- [Usage](#usage)
- [Field Types](#field-types)
- [API Services](#api-services)
- [Architecture](#architecture)
- [Error Handling](#error-handling)
- [Security](#security)
- [Troubleshooting](#troubleshooting)
- [Contributing](#contributing)
- [Sponsorship](#sponsorship)

## Features

- **Secure API Integration**: Secure API key management using Drupal's Key module
- **Customer Reference Field**: Advanced Tagify-based field widget for customer references
- **Real-time Search**: Auto-complete functionality with live Twenty CRM data
- **Multi-entity Support**: Contact and company management
- **Configurable Endpoints**: Support for different Twenty CRM instances
- **Bearer Token Authentication**: Secure OAuth 2.0 compliant authentication
- **Comprehensive Logging**: Detailed error handling and logging
- **Type-safe Operations**: Built on standalone PHP package with DTOs
- **Drupal Integration**: Native Drupal services and configuration management

## Requirements

- **Drupal**: 9.5+ or 10+ or 11+
- **PHP**: 8.1 or higher
- **Required Modules**:
  - [Key](https://www.drupal.org/project/key) - Secure API key storage
  - [Tagify](https://www.drupal.org/project/tagify) - Advanced tagging interface
- **PHP Package**: factorial-io/twenty-crm-php-client (installed via Composer)

## Installation

### Via Composer (Recommended)

```bash
composer require drupal/twenty_crm
drush en twenty_crm
```

### Manual Installation

1. Download and place this module in `web/modules/contrib/twenty_crm/`
2. Install dependencies: `composer require factorial-io/twenty-crm-php-client`
3. Enable the module: `drush en twenty_crm`
4. Configure API credentials (see Configuration section)

## Configuration

### 1. API Key Setup

1. Navigate to **Configuration > System > Keys** (`/admin/config/system/keys`)
2. Click **Add key**
3. Set the following:
   - **Key ID**: `twenty_crm_api_key`
   - **Key name**: Twenty CRM API Key
   - **Key type**: Authentication
   - **Key provider**: Configuration
   - **Key value**: Your Twenty CRM API token

### 2. Module Configuration

Access the settings form via:
- **Configuration > Web Services > Twenty CRM** (`/admin/config/services/twenty-crm`)
- Module configuration link at `/admin/modules`

Configure the following:
- **Base URL**: Your Twenty CRM instance URL (e.g., `https://api.twenty.com/rest/`)
- **API Key**: Select the key created in step 1
- **Default Settings**: Timeout, cache settings, etc.

## Usage

### Customer Reference Field

The module provides a custom field type for referencing Twenty CRM customers:

1. **Add Field**: Go to any content type and add a **Customer Reference** field
2. **Configure Widget**: The field uses Tagify for auto-complete functionality
3. **Search & Select**: Users can search and select customers directly from Twenty CRM

### Programmatic Usage

#### Contact Service

```php
use Factorial\TwentyCrm\DTO\ContactSearchFilter;
use Factorial\TwentyCrm\DTO\SearchOptions;

$contactService = \Drupal::service('twenty_crm.contact_service');

// Search contacts
$filter = new ContactSearchFilter(
  email: 'john@example.com',
  name: 'John Doe'
);

$options = new SearchOptions(
  limit: 10,
  orderBy: 'name.firstName'
);

$contacts = $contactService->findContacts($filter, $options);

// Get specific contact
$contact = $contactService->getContactById('contact-uuid');
```

#### Company Service

```php
$companyService = \Drupal::service('twenty_crm.company_service');

// Search companies
$companies = $companyService->findCompanies(
  new CompanySearchFilter(name: 'Acme Corp')
);
```

#### Direct Client Access

```php
$client = \Drupal::service('twenty_crm.client');
$contacts = $client->contacts()->find($filter, $options);
$companies = $client->companies()->find($companyFilter);
```

## Field Types

### Customer Reference Field

- **Field Type**: `customer_reference`
- **Widget**: Tagify-based auto-complete
- **Formatter**: Displays customer name with optional link
- **Storage**: Stores Twenty CRM customer UUID
- **Features**:
  - Real-time search
  - Multiple customer selection
  - Cached results for performance
  - Validation against Twenty CRM data

## API Services

The module provides several Drupal services:

- `twenty_crm.client` - Core Twenty CRM client
- `twenty_crm.contact_service` - Contact management
- `twenty_crm.company_service` - Company management
- `twenty_crm.client_factory` - Client factory with configuration

## Architecture

### PHP Client Package

Built on [`factorial-io/twenty-crm-php-client`](https://github.com/factorial-io/twenty-crm-php-client):
- Framework-agnostic API client
- PSR-18 HTTP client abstraction
- PSR-3 logging support
- Type-safe DTOs and interfaces
- Comprehensive error handling

### Drupal Integration Layer

- **Services**: Native Drupal service integration
- **Configuration**: Drupal configuration system
- **Security**: Key module integration
- **Caching**: Drupal cache integration
- **Logging**: Drupal logging system

## Error Handling

The module provides comprehensive error handling:

- **API Key Validation**: Ensures valid authentication
- **HTTP Error Handling**: Graceful handling of network issues
- **JSON Parsing**: Robust data parsing with fallbacks
- **User Feedback**: Clear error messages in the UI
- **Logging**: Detailed logs in the `twenty_crm` channel

### Debugging

Check logs at **Reports > Recent log messages** (`/admin/reports/dblog`) and filter by `twenty_crm`.

## Security

- **API Keys**: Securely stored using Drupal's Key module
- **HTTPS**: All requests use HTTPS by default
- **Authentication**: Bearer token authentication following OAuth 2.0 standards
- **Validation**: Input validation on all API calls
- **Permissions**: Drupal permission system integration

## Troubleshooting

### Common Issues

1. **API Key Not Found**
   - Ensure key is created with ID `twenty_crm_api_key`
   - Verify key has the correct API token

2. **Connection Issues**
   - Check base URL configuration
   - Verify network connectivity
   - Check firewall settings

3. **Tagify Not Loading**
   - Ensure Tagify module is enabled
   - Clear cache: `drush cr`
   - Check JavaScript console for errors

### Support

- **Issue Queue**: [Drupal.org issue queue](https://www.drupal.org/project/issues/twenty_crm)
- **Documentation**: [Project page](https://www.drupal.org/project/twenty_crm)

## Contributing

We welcome contributions! Please:

1. Check the [issue queue](https://www.drupal.org/project/issues/twenty_crm) for existing issues
2. Follow [Drupal coding standards](https://www.drupal.org/docs/develop/standards)
3. Create merge requests following [GitLab contribution guidelines](https://www.drupal.org/drupalorg/docs/gitlab-integration/contributing)
4. Test your changes thoroughly

## Sponsorship

Development of this module is sponsored by **[Factorial.io](https://www.factorial.io)** - a leading Drupal development agency specializing in enterprise web solutions and API integrations.

Factorial.io provides expert Drupal consulting, custom development, and digital transformation services. Visit [factorial.io](https://www.factorial.io) to learn more about their services and how they can help with your next project.

---

**Maintainers**: Factorial.io Development Team  
**License**: GPL-2.0-or-later  
**Project Page**: https://www.drupal.org/project/twenty_crm