<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\GraphQLCompose\FieldType;

use Drupal\Core\Field\FieldItemInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerItemInterface;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerTrait;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeFieldTypeBase;
use Drupal\twenty_crm\Plugin\Field\FieldType\CustomerReferenceItem;
use Drupal\twenty_crm\TwentyCrmContactService;

/**
 * {@inheritDoc}
 *
 * @GraphQLComposeFieldType(
 *   id = "customer_reference",
 *   type_sdl = "TwentyCrmCustomerReference",
 * )
 */
class CustomerReferenceFieldType extends GraphQLComposeFieldTypeBase implements FieldProducerItemInterface {

  use FieldProducerTrait;

  /**
   * {@inheritdoc}
   */
  public function resolveFieldItem(FieldItemInterface $item, FieldContext $context) {
    if ($item instanceof CustomerReferenceItem) {
      $data = [
        'contactUuid' => $item->contact_uuid,
        'companyUuid' => $item->company_uuid,
        'companyName' => $item->company_name,
        'contactName' => $item->contact_name,
      ];

      // If we have a contact UUID, fetch additional contact data from the CRM
      if (!empty($item->contact_uuid)) {
        $data['contact'] = $this->getContactData($item->contact_uuid);
      }

      // If we have a company UUID, reuse the company data method from CompanyReferenceFieldType
      if (!empty($item->company_uuid)) {
        $data['company'] = CompanyReferenceFieldType::getCompanyData($item->company_uuid);
      }

      return $data;
    }
  }

  /**
   * Get contact data from Twenty CRM API.
   *
   * @param string $contactUuid
   *   The contact UUID.
   *
   * @return array|null
   *   The contact data or NULL if not found.
   */
  private function getContactData(string $contactUuid): ?array {
    /** @var \Drupal\twenty_crm\TwentyCrmContactService $contactService */
    $contactService = \Drupal::service('twenty_crm.contact_service');
    $contact = $contactService->getContactById($contactUuid);
    
    if (!$contact) {
      return null;
    }

    return [
      'id' => $contact->getId(),
      'email' => $contact->getEmail(),
      'firstName' => $contact->getFirstName(),
      'lastName' => $contact->getLastName(),
      'fullName' => $contact->getFullName(),
      'phone' => $contact->getPhone(),
      'jobTitle' => $contact->getJobTitle(),
      'companyId' => $contact->getCompanyId(),
      'customFields' => json_encode($contact->getCustomFields()),
      'createdAt' => $contact->getCreatedAt()?->format('c'),
      'updatedAt' => $contact->getUpdatedAt()?->format('c'),
    ];
  }

}