<?php

declare(strict_types=1);

namespace Drupal\twenty_crm;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use Factorial\TwentyCrm\Auth\BearerTokenAuth;
use Factorial\TwentyCrm\Client\ClientInterface;
use Factorial\TwentyCrm\Client\TwentyCrmClient;
use Factorial\TwentyCrm\Http\GuzzleHttpClient;
use GuzzleHttp\Psr7\HttpFactory;
use Psr\Http\Client\ClientInterface as HttpClientInterface;
use Psr\Log\LoggerInterface;

/**
 * Factory for creating Twenty CRM client instances.
 */
final class TwentyCrmClientFactory {

  public const CONFIG_NAME = 'twenty_crm.settings';
  public const DEFAULT_API_KEY_ID = 'twenty_crm_api_key';

  /**
   * Constructs a TwentyCrmClientFactory object.
   */
  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly KeyRepositoryInterface $keyRepository,
    protected readonly HttpClientInterface $httpClient,
    protected readonly LoggerInterface $logger,
  ) {}

  /**
   * Create a Twenty CRM client instance.
   *
   * @return \Factorial\TwentyCrm\Client\ClientInterface
   *   The configured client instance.
   *
   * @throws \RuntimeException
   *   When configuration is invalid or API key is not found.
   */
  public function create(): ClientInterface {
    $config = $this->configFactory->get(self::CONFIG_NAME);
    $baseUrl = $config->get('base_url');

    if (empty($baseUrl)) {
      throw new \RuntimeException('Twenty CRM base URL is not configured');
    }

    // Get API key.
    $keyId = $config->get('api_key_id') ?? self::DEFAULT_API_KEY_ID;
    $key = $this->keyRepository->getKey($keyId);

    if (!$key) {
      throw new \RuntimeException("Twenty CRM API key '{$keyId}' not found");
    }

    $apiToken = $key->getKeyValue();
    if (empty($apiToken)) {
      throw new \RuntimeException("Twenty CRM API key '{$keyId}' is empty");
    }

    // Create HTTP client components using Drupal's HTTP client.
    $httpFactory = new HttpFactory();
    $auth = new BearerTokenAuth($apiToken);

    // Create HTTP client wrapper.
    $httpClientWrapper = new GuzzleHttpClient(
      $this->httpClient,
      $httpFactory,
      $httpFactory,
      $auth,
      $baseUrl,
      $this->logger
    );

    // Create and return the main client.
    return new TwentyCrmClient($httpClientWrapper);
  }

}
