<?php

declare(strict_types=1);

namespace Drupal\twenty_crm;

use Drupal\Core\Logger\LoggerChannelInterface;
use Factorial\TwentyCrm\Client\ClientInterface;
use Factorial\TwentyCrm\DTO\Contact;
use Factorial\TwentyCrm\DTO\ContactCollection;
use Factorial\TwentyCrm\DTO\FilterInterface;
use Factorial\TwentyCrm\DTO\SearchOptions;

/**
 * Drupal wrapper for Twenty CRM Contact service.
 */
final class TwentyCrmContactService {

  /**
   * Constructs a TwentyCrmContactService object.
   */
  public function __construct(
    protected readonly ClientInterface $client,
    protected readonly LoggerChannelInterface $logger,
  ) {}

  /**
   * Find contacts/people based on search criteria.
   *
   * @param \Factorial\TwentyCrm\DTO\FilterInterface $filter
   *   The search filter criteria.
   * @param \Factorial\TwentyCrm\DTO\SearchOptions $options
   *   The search options (pagination, sorting, etc.).
   *
   * @return ContactCollection|null
   *   The contact collection with results, or NULL on failure.
   */
  public function findContacts(FilterInterface $filter, SearchOptions $options): ?ContactCollection {
    try {
      $this->logger->info('Twenty CRM: Searching contacts', [
        'has_filters' => $filter->hasFilters(),
        'limit' => $options->limit,
      ]);

      return $this->client->contacts()->find($filter, $options);

    }
    catch (\Exception $e) {
      $this->logger->error('Twenty CRM contact search failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Get a contact by ID.
   *
   * @param string $id
   *   The contact ID.
   *
   * @return Contact|null
   *   The contact or NULL if not found.
   */
  public function getContactById(string $contact_id): ?Contact {
    try {
      return $this->client->contacts()->getById($contact_id);
    }
    catch (\Exception $e) {
      $this->logger->error('Twenty CRM get contact by ID failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
