<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Controller;

use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\twenty_crm\TwentyCrmCompanyService;
use Drupal\twenty_crm\TwentyCrmContactService;
use Factorial\TwentyCrm\DTO\CompanySearchFilter;
use Factorial\TwentyCrm\DTO\ContactSearchFilter;
use Factorial\TwentyCrm\DTO\SearchOptions;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Twenty CRM autocomplete functionality.
 */
class AutocompleteController extends ControllerBase {

  /**
   * Constructs an AutocompleteController object.
   */
  public function __construct(
    private readonly TwentyCrmCompanyService $companyService,
    private readonly TwentyCrmContactService $contactService,
  ) {}

  /**
   * Company autocomplete callback.
   */
  public function companyAutocomplete(Request $request): CacheableJsonResponse {
    $query = $request->query->get('q', '');

    if (\strlen($query) < 2) {
      return $this->createCachedResponse([]);
    }

    $filter = new CompanySearchFilter(name: $query);

    $options = new SearchOptions(limit: 10);

    $results = $this->companyService->findCompanies($filter, $options);

    if (!$results || $results->isEmpty()) {
      return $this->createCachedResponse([], ['twenty_crm:company_search']);
    }

    $suggestions = [];

    foreach ($results->getCompanies() as $company) {
      $location = $company->getLocationString();
      $location = !empty($location) ? ' (' . $location . ')' : '';

      $suggestions[] = [
        'uuid' => $company->getId(),
        'name' => $company->getName(),
        'location' => $location,
        'city' => $company->getAddressCity() ?? '',
        'country' => $company->getAddressCountry() ?? '',
      ];
    }

    return $this->createCachedResponse($suggestions, ['twenty_crm:company_search']);
  }

  /**
   * Contact autocomplete callback.
   */
  public function contactAutocomplete(Request $request, string $company_uuid = ''): CacheableJsonResponse {
    $query = $request->query->get('q', '');

    if (\strlen($query) < 2 || $company_uuid === '__none__' || empty($company_uuid)) {
      return $this->createCachedResponse([]);
    }

    // Search by first name within the specified company.
    $filter = new ContactSearchFilter(
      firstName: $query,
      companyId: $company_uuid
    );

    $options = new SearchOptions(limit: 10);

    $results = $this->contactService->findContacts($filter, $options);

    if (!$results || $results->isEmpty()) {
      return $this->createCachedResponse([], ['twenty_crm:contact_search']);
    }

    $suggestions = [];

    foreach ($results->getContacts() as $contact) {
      $name = $contact->getFullName();

      if (empty($name)) {
        $name = $contact->getEmail() ?? 'Unknown';
      }

      $suggestions[] = [
        'uuid' => $contact->getId(),
        'name' => $name,
        'email' => $contact->getEmail() ?? '',
        'jobTitle' => $contact->getJobTitle() ?? '',
      ];
    }

    return $this->createCachedResponse($suggestions, ['twenty_crm:contact_search']);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('twenty_crm.company_service'),
      $container->get('twenty_crm.contact_service'),
    );
  }

  /**
   * Create a cached JSON response.
   */
  private function createCachedResponse(array $data, array $cache_tags = []): CacheableJsonResponse {
    $response = new CacheableJsonResponse($data);
    $cache_metadata = new CacheableMetadata();
    $cache_metadata->setCacheMaxAge(15);
    $cache_metadata->setCacheContexts(['url.query_args:q', 'url.path']);
    if (!empty($cache_tags)) {
      $cache_metadata->setCacheTags($cache_tags);
    }
    $response->addCacheableDependency($cache_metadata);
    return $response;
  }

}
