<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\Field\FieldFormatter;

use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\twenty_crm\TwentyCrmCompanyService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'company_reference_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "company_reference_formatter",
 *   label = @Translation("Company Reference Formatter"),
 *   field_types = {
 *     "company_reference"
 *   }
 * )
 */
class CompanyReferenceFormatter extends FormatterBase {

  /**
   * Constructs a CompanyReferenceFormatter object.
   */
  public function __construct(
    string $plugin_id,
    mixed $plugin_definition,
    mixed $field_definition,
    array $settings,
    string $label,
    string $view_mode,
    array $third_party_settings,
    private readonly TwentyCrmCompanyService $companyService,
    private readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('twenty_crm.company_service'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'display_format' => 'name_location',
      'link_to_crm' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['display_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#default_value' => $this->getSetting('display_format'),
      '#options' => [
        'name_only' => $this->t('Company name only'),
        'name_location' => $this->t('Company name with location'),
        'full' => $this->t('Full information'),
      ],
    ];

    $elements['link_to_crm'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Link to Twenty CRM'),
      '#default_value' => $this->getSetting('link_to_crm'),
      '#description' => $this->t('Make company names clickable links to the Twenty CRM system.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $display_format_options = [
      'name_only' => $this->t('Company name only'),
      'name_location' => $this->t('Company name with location'),
      'full' => $this->t('Full information'),
    ];

    $summary[] = $this->t('Display: @format', ['@format' => $display_format_options[$this->getSetting('display_format')]]);

    if ($this->getSetting('link_to_crm')) {
      $summary[] = $this->t('Links to Twenty CRM');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      if (empty($item->company_uuid)) {
        continue;
      }

      $display_format = $this->getSetting('display_format');
      $link_to_crm = $this->getSetting('link_to_crm');

      // Use stored values if available, otherwise try to fetch from CRM.
      $company_name = $item->company_name;
      $company_city = $item->company_city;
      $company_country = $item->company_country;

      // If we don't have cached display name, try to fetch from CRM.
      if (empty($company_name)) {
        $company = $this->companyService->getCompanyById($item->company_uuid);
        if ($company) {
          $company_name = $company->getName() ?? 'Unknown Company';
          $company_city = $company->getAddressCity() ?? '';
          $company_country = $company->getAddressCountry() ?? '';
        }
      }

      if (empty($company_name)) {
        $company_name = 'Unknown Company';
      }

      // Build display text based on format setting.
      $display_text = match ($display_format) {
        'name_only' => $company_name,
        'name_location' => $this->buildNameLocationDisplay($company_name, $company_city, $company_country),
        'full' => $this->buildFullDisplay($company_name, $company_city, $company_country),
        default => $company_name,
      };

      if ($link_to_crm && $this->getTwentyCrmUrl()) {
        $elements[$delta] = [
          '#type' => 'link',
          '#title' => $display_text,
          '#url' => $this->buildTwentyCrmUrl($item->company_uuid),
          '#attributes' => [
            'target' => '_blank',
            'rel' => 'noopener noreferrer',
          ],
        ];
      }
      else {
        $elements[$delta] = [
          '#markup' => $display_text,
        ];
      }
    }

    return $elements;
  }

  /**
   * Build name with location display format.
   */
  private function buildNameLocationDisplay(string $company_name, ?string $city, ?string $country): string {
    $location_parts = [];
    if (!empty($city)) {
      $location_parts[] = $city;
    }
    if (!empty($country)) {
      $location_parts[] = $country;
    }
    
    if (!empty($location_parts)) {
      return $company_name . ' (' . implode(', ', $location_parts) . ')';
    }
    
    return $company_name;
  }

  /**
   * Build full display format with all available information.
   */
  private function buildFullDisplay(string $company_name, ?string $city, ?string $country): string {
    $parts = [$company_name];
    
    if (!empty($city)) {
      $parts[] = $city;
    }
    if (!empty($country)) {
      $parts[] = $country;
    }

    return implode(' - ', $parts);
  }

  /**
   * Get the Twenty CRM base URL from configuration.
   */
  private function getTwentyCrmUrl(): ?string {
    $config = $this->configFactory->get('twenty_crm.settings');
    return $config->get('base_url');
  }

  /**
   * Build Twenty CRM URL for a company.
   */
  private function buildTwentyCrmUrl(string $company_uuid): Url {
    $base_url = $this->getTwentyCrmUrl();
    if (!$base_url) {
      return Url::fromRoute('<nolink>');
    }

    // Convert REST API URL to frontend URL
    // From: https://factorial.twenty.com/rest
    // To: https://factorial.twenty.com/object/company/{uuid}
    $frontend_url = str_replace('/rest', '', $base_url);
    $company_url = rtrim($frontend_url, '/') . '/object/company/' . $company_uuid;

    return Url::fromUri($company_url);
  }

}