<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\Field\FieldType;

use Drupal\Component\Utility\Random;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'company_reference' field type.
 *
 * @FieldType(
 *   id = "company_reference",
 *   label = @Translation("Twenty CRM Company Reference"),
 *   default_formatter = "company_reference_formatter",
 *   default_widget = "company_reference_widget",
 *   description = @Translation("A reference to a Twenty CRM company")
 * )
 */
class CompanyReferenceItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName() {
    return 'company_uuid';
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings() {
    return [
      'max_length' => 36,
      'is_ascii' => TRUE,
      'case_sensitive' => FALSE,
    ] + parent::defaultStorageSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['company_uuid'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Company UUID'))
      ->setSetting('case_sensitive', $field_definition->getSetting('case_sensitive'))
      ->setRequired(FALSE);

    $properties['company_name'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Company Name'))
      ->setRequired(FALSE);

    $properties['company_city'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Company City'))
      ->setRequired(FALSE);

    $properties['company_country'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Company Country'))
      ->setRequired(FALSE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = [
      'columns' => [
        'company_uuid' => [
          'type' => 'varchar_ascii',
          'length' => 36,
          'not null' => FALSE,
        ],
        'company_name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
        'company_city' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
        'company_country' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'company_uuid' => ['company_uuid'],
      ],
    ];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition) {
    $random = new Random();
    $values['company_uuid'] = $random->name(8, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(12, TRUE);
    $values['company_name'] = $random->sentences(2);
    $values['company_city'] = $random->name();
    $values['company_country'] = $random->name();
    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('company_uuid')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public function getConstraints() {
    $constraints = parent::getConstraints();
    return $constraints;
  }

}