<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\GraphQLCompose\FieldType;

use Drupal\Core\Field\FieldItemInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerItemInterface;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerTrait;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeFieldTypeBase;
use Drupal\twenty_crm\Plugin\Field\FieldType\CompanyReferenceItem;
use Drupal\twenty_crm\TwentyCrmCompanyService;

/**
 * {@inheritDoc}
 *
 * @GraphQLComposeFieldType(
 *   id = "company_reference",
 *   type_sdl = "TwentyCrmCompanyReference",
 * )
 */
class CompanyReferenceFieldType extends GraphQLComposeFieldTypeBase implements FieldProducerItemInterface {

  use FieldProducerTrait;

  /**
   * {@inheritdoc}
   */
  public function resolveFieldItem(FieldItemInterface $item, FieldContext $context) {
    if ($item instanceof CompanyReferenceItem) {
      $data = [
        'companyUuid' => $item->company_uuid,
        'companyName' => $item->company_name,
        'companyCity' => $item->company_city,
        'companyCountry' => $item->company_country,
      ];

      // If we have a UUID, fetch additional data from the CRM
      if (!empty($item->company_uuid)) {
        $data['company'] = $this->getCompanyData($item->company_uuid);
      }

      return $data;
    }
  }

  /**
   * Get company data from Twenty CRM API.
   *
   * @param string $companyUuid
   *   The company UUID.
   *
   * @return array|null
   *   The company data or NULL if not found.
   */
  public static function getCompanyData(string $companyUuid): ?array {
    /** @var \Drupal\twenty_crm\TwentyCrmCompanyService $companyService */
    $companyService = \Drupal::service('twenty_crm.company_service');
    $company = $companyService->getCompanyById($companyUuid);
    
    if (!$company) {
      return null;
    }

    return [
      'id' => $company->getId(),
      'name' => $company->getName(),
      'domainName' => self::formatLinkCollection($company->getDomainName()),
      'facebook' => self::formatLinkCollection($company->getFacebook()),
      'linkedinLink' => self::formatLinkCollection($company->getLinkedinLink()),
      'xLink' => self::formatLinkCollection($company->getXLink()),
      'addressCity' => $company->getAddressCity(),
      'addressCountry' => $company->getAddressCountry(),
      'addressStreet1' => $company->getAddressStreet1(),
      'addressStreet2' => $company->getAddressStreet2(),
      'addressState' => $company->getAddressState(),
      'addressPostcode' => $company->getAddressPostcode(),
      'employees' => $company->getEmployees(),
      'locationString' => $company->getLocationString(),
      'customFields' => json_encode($company->getCustomFields()),
      'createdAt' => $company->getCreatedAt()?->format('c'),
      'updatedAt' => $company->getUpdatedAt()?->format('c'),
    ];
  }

  /**
   * Format a LinkCollection for GraphQL output.
   *
   * @param \Factorial\TwentyCrm\DTO\LinkCollection|null $linkCollection
   *   The link collection to format.
   *
   * @return array|null
   *   The formatted link collection or NULL.
   */
  private static function formatLinkCollection($linkCollection): ?array {
    if (!$linkCollection || $linkCollection->isEmpty()) {
      return null;
    }

    return [
      'primaryLink' => $linkCollection->getPrimaryLink() ? [
        'url' => $linkCollection->getPrimaryLink()->getUrl(),
        'label' => $linkCollection->getPrimaryLink()->getLabel(),
      ] : null,
      'secondaryLinks' => array_map(fn($link) => [
        'url' => $link->getUrl(),
        'label' => $link->getLabel(),
      ], $linkCollection->getSecondaryLinks()),
    ];
  }

}