<?php

declare(strict_types=1);

namespace Drupal\twenty_crm;

use Factorial\TwentyCrm\Client\ClientInterface;
use Factorial\TwentyCrm\DTO\Company;
use Factorial\TwentyCrm\DTO\CompanyCollection;
use Factorial\TwentyCrm\DTO\FilterInterface;
use Factorial\TwentyCrm\DTO\SearchOptions;

/**
 * Drupal wrapper service for Twenty CRM company operations.
 */
class TwentyCrmCompanyService {

  /**
   * Constructs a TwentyCrmCompanyService object.
   *
   * @param \Factorial\TwentyCrm\Client\ClientInterface $client
   *   The Twenty CRM client.
   */
  public function __construct(
    private readonly ClientInterface $client,
  ) {}

  /**
   * Find companies using the Twenty CRM API.
   *
   * @param \Factorial\TwentyCrm\DTO\FilterInterface $filter
   *   The search filter criteria.
   * @param \Factorial\TwentyCrm\DTO\SearchOptions $options
   *   The search options.
   *
   * @return \Factorial\TwentyCrm\DTO\CompanyCollection|null
   *   The search results or NULL on error.
   */
  public function findCompanies(FilterInterface $filter, SearchOptions $options): ?CompanyCollection {
    try {
      return $this->client->companies()->find($filter, $options);
    }
    catch (\Exception $e) {
      \Drupal::logger('twenty_crm')->error('Company search failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Get a company by UUID.
   *
   * @param string $id
   *   The company UUID.
   *
   * @return \Factorial\TwentyCrm\DTO\Company|null
   *   The company or NULL if not found/error.
   */
  public function getCompanyById(string $id): ?Company {
    try {
      return $this->client->companies()->getById($id);
    }
    catch (\Exception $e) {
      \Drupal::logger('twenty_crm')->error('Company fetch by ID failed for @id: @message', [
        '@id' => $id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
