<?php

namespace Drupal\typed_entity\Render;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\typed_entity\TypedEntityContext;
use Drupal\typed_entity\Annotation\VariantInterface;
use Drupal\typed_entity\WrappedEntities\WrappedEntityInterface;

/**
 * Renderers are classes that encapsulate buisiness logic rendering an entity.
 */
interface TypedEntityRendererInterface extends VariantInterface, ContainerInjectionInterface {

  /**
   * Returns a render array representation of the wrapped entity.
   *
   * @param \Drupal\typed_entity\WrappedEntities\WrappedEntityInterface $wrapped_entity
   *   The wrapped entity to render.
   * @param \Drupal\typed_entity\TypedEntityContext $context
   *   The context this entity is rendered in. This contains arbitrary
   *   information on how to render the entity. Special keys:
   *     - 'view_mode': The view mode to use to render the entity. Leave it
   *       empty to use a static value declared in the renderer.
   *
   * @return array
   *   A render array.
   */
  public function build(WrappedEntityInterface $wrapped_entity, TypedEntityContext $context): array;

  /**
   * Alter the render array for the associated entity.
   *
   * The children added here will be rendered without any changes necessary in
   * the template. If you want to pass raw variables to the template use
   * ::preprocess.
   *
   * @param array $build
   *   The render array being preprocessed.
   * @param \Drupal\typed_entity\WrappedEntities\WrappedEntityInterface $wrapped_entity
   *   The wrapped entity.
   * @param \Drupal\Core\Entity\Display\EntityViewDisplayInterface $display
   *   The entity view display.
   *
   * @see ::preprocess()
   */
  public function viewAlter(array &$build, WrappedEntityInterface $wrapped_entity, EntityViewDisplayInterface $display): void;

  /**
   * Alter the render array generated by an EntityDisplay for an entity.
   *
   * @param array $build
   *   The renderable array generated by the EntityDisplay.
   * @param \Drupal\typed_entity\WrappedEntities\WrappedEntityInterface $wrapped_entity
   *   The wrapped entity.
   * @param array $context
   *   An associative array containing:
   *   - entity: The entity being rendered.
   *   - view_mode: The view mode; for example, 'full' or 'teaser'.
   *   - display: The EntityDisplay holding the display options.
   */
  public function displayBuildAlter(array &$build, WrappedEntityInterface $wrapped_entity, array $context): void;

  /**
   * Alter entity renderable values before cache checking during rendering.
   *
   * The values in the #cache key of renderable arrays are used to determine if
   * a cache entry exists for the entity's rendered output. Ideally only values
   * that pertain to caching should be altered in this hook.
   *
   * @param array &$build
   *   A renderable array containing the entity's caching and view mode values.
   * @param \Drupal\typed_entity\WrappedEntities\WrappedEntityInterface $wrapped_entity
   *   The wrapped entity that is being viewed.
   * @param string $view_mode
   *   The view_mode that is to be used to display the entity.
   */
  public function buildDefaultsAlter(array &$build, WrappedEntityInterface $wrapped_entity, string $view_mode): void;

  /**
   * Custom preprocessing for the renderer.
   *
   * @param array $variables
   *   The render array passed by reference.
   * @param \Drupal\typed_entity\WrappedEntities\WrappedEntityInterface $wrapped_entity
   *   The typed entity being processed. It is only here for context. It is not
   *   recommended to extract data from the entity directly without passing it
   *   through the render pipeline.
   */
  public function preprocess(array &$variables, WrappedEntityInterface $wrapped_entity): void;

}
