<?php

declare(strict_types=1);

namespace Drupal\typesense_integration\Plugin\search_api\processor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Plugin\DataType\EntityAdapter;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Drupal\search_api\Datasource\DatasourceInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\search_api\Processor\ProcessorProperty;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Index the URL for an entity.
 *
 * @SearchApiProcessor(
 *   id = "typesense_entity_url",
 *   label = @Translation("Entity URL"),
 *   description = @Translation("The canonical URL of an entity."),
 *   stages = {
 *     "add_properties" = 0,
 *   },
 *   locked = true,
 *   hidden = true,
 * )
 */
class EntityUrl extends ProcessorPluginBase {

  /**
   * The file URL generator.
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var static $plugin */
    $plugin = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $plugin->setFileUrlGenerator($container->get('file_url_generator'));

    return $plugin;
  }

  /**
   * Retrieves the file url generator service.
   *
   * @return \Drupal\Core\File\FileUrlGeneratorInterface
   *   The file url generator service.
   */
  public function getFileUrlGenerator(): FileUrlGeneratorInterface {
    /* @phpstan-ignore-next-line */
    return $this->fileUrlGenerator ?: \Drupal::service('file_url_generator');
  }

  /**
   * Sets the file url generator.
   *
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   The file url generator.
   *
   * @return $this
   */
  public function setFileUrlGenerator(FileUrlGeneratorInterface $fileUrlGenerator) {
    $this->fileUrlGenerator = $fileUrlGenerator;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPropertyDefinitions(?DatasourceInterface $datasource = NULL): array {
    $properties = [];

    if (!$datasource) {
      $pluginDefinition = $this->getPluginDefinition();
      $definition = [
        'label' => $pluginDefinition['label'],
        'description' => $pluginDefinition['description'],
        'type' => 'string',
        'is_list' => FALSE,
        'processor_id' => $this->getPluginId(),
      ];
      $properties['entity_url'] = new ProcessorProperty($definition);
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function addFieldValues(ItemInterface $item): void {
    $object = $item->getOriginalObject();
    if (!$object instanceof EntityAdapter) {
      return;
    }
    $entity = $object->getEntity();

    if (!$entity instanceof ContentEntityInterface) {
      return;
    }

    $url = $this->getEntityUrl($entity);

    if (!$url) {
      return;
    }

    $fields = $item->getFields(FALSE);
    $fields = $this->getFieldsHelper()->filterForPropertyPath($fields, NULL, 'entity_url');
    $field = reset($fields);
    $field->addValue($url->toString());
  }

  /**
   * Get the URL for an entity.
   *
   * @return \Drupal\Core\Url|null
   *   The URL.
   */
  private function getEntityUrl(ContentEntityInterface $entity): ?Url {
    if ($entity instanceof Media) {
      $fid = $entity->getSource()->getSourceFieldValue($entity);
      $file = File::load($fid);
      if ($file) {
        $generator = $this->getFileUrlGenerator();
        $uri = $file->getFileUri();
        return $generator->generate($uri);
      }
      return NULL;
    }
    return $entity->toUrl('canonical');
  }

}
