<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\EventSubscriber;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\search_api_typesense\Event\TypesenseSchemaEvent;
use Drupal\search_api_typesense\Event\TypesenseSchemaEvents;
use Drupal\typesense_graphql\TypesenseSchemaAwareInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for TypeSense schema.
 */
class TypesenseSchemaSubscriber implements EventSubscriberInterface {

  /**
   * The Search API data type plugin manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $dataTypePluginManager;

  /**
   * Constructs a new TypesenseSchemaSubscriber.
   *
   * @param \Drupal\Component\Plugin\PluginManagerInterface $data_type_plugin_manager
   *   The Search API data type plugin manager.
   */
  public function __construct(PluginManagerInterface $data_type_plugin_manager) {
    $this->dataTypePluginManager = $data_type_plugin_manager;
  }

  /**
   * Alters the generated Typesense schema.
   *
   * @param \Drupal\search_api_typesense\Event\TypesenseSchemaEvent $event
   *   The event.
   */
  public function onAlterSchema(TypesenseSchemaEvent $event) {
    $schema = $event->getGeneratedSchema();

    // This allows us to index objects.
    $schema['enable_nested_fields'] = TRUE;

    $objectFields = [];

    foreach ($schema['fields'] as $delta => $field) {
      $type = $field['type'] ?? NULL;
      if (!$type) {
        continue;
      }

      // Check if a plugin exists for this field type.
      $type = 'typesense_' . $type;
      if ($this->dataTypePluginManager->hasDefinition($type)) {
        try {
          $plugin = $this->dataTypePluginManager->createInstance($type);

          // Check if the plugin implements TypesenseSchemaAware interface.
          if ($plugin instanceof TypesenseSchemaAwareInterface) {
            $schema['fields'][$delta] = $plugin->alterSchemaField($field);
            // Add any additional fields that the plugin wants to include.
            $additionalFields = $plugin->addAdditionalFields($field);
            if (!empty($additionalFields)) {
              $objectFields = array_merge($objectFields, $additionalFields);
            }
            continue;
          }
        }
        catch (\Exception $e) {
        }
      }

      // Handle other field types that don't have plugins or don't implement
      // TypesenseSchemaAwareInterface.
      if ($type === 'float[]') {
        // This needs to be in sync with the actual size of the embeddings we
        // load.
        $schema['fields'][$delta]['num_dim'] = 1536;
      }
    }

    $schema['fields'] = array_merge($schema['fields'], $objectFields);

    $event->setGeneratedSchema($schema);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      TypesenseSchemaEvents::ALTER_SCHEMA => 'onAlterSchema',
    ];
  }

}
