<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Model;

/**
 * Helper object for defining a fulltext field.
 *
 * @see https://typesense.org/docs/28.0/api/search.html#query-parameters
 */
class TypesenseQueryFulltextField {

  /**
   * The query weight.
   *
   * @var int
   */
  public int $weight = 1;

  /**
   * Whether prefix search should be enabled for this field.
   *
   * @var bool
   */
  public bool $prefix = FALSE;

  /**
   * Whether to enable infix search.
   *
   * @var 'off' | 'always' | 'fallback'
   */
  public string $infix = 'off';

  /**
   * Number of typos.
   *
   * @var int
   */
  public int $numTypos = 0;

  /**
   * Constructs a new TypesenseQueryFulltextField object.
   *
   * @param string $name
   *   The name of the field.
   */
  public function __construct(
    public readonly string $name,
  ) {
  }

  /**
   * Set the weight.
   *
   * @param int $weight
   *   The weight.
   *
   * @return static
   */
  public function weight(int $weight): static {
    $this->weight = $weight;
    return $this;
  }

  /**
   * Enable prefix search.
   *
   * @return static
   */
  public function prefix(): static {
    $this->prefix = TRUE;
    return $this;
  }

  /**
   * Enable infix search.
   *
   * @param 'always'|'fallback' $infix
   *   The infix type.
   *
   * @return static
   */
  public function infix(string $infix = 'always'): static {
    $this->infix = $infix;
    return $this;
  }

  /**
   * Set the number of typos allowed.
   *
   * @param int $numTypos
   *   The number of typos.
   *
   * @return static
   */
  public function numTypos(int $numTypos): static {
    $this->numTypos = $numTypos;
    return $this;
  }

}
