<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql;

use Drupal\graphql_core_schema\EntitySchemaHelper;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_typesense\Plugin\search_api\backend\SearchApiTypesenseBackend;
use GraphQL\Type\Definition\CustomScalarType;
use GraphQL\Type\Definition\InterfaceType;
use GraphQL\Type\Definition\ObjectType;
use GraphQL\Type\Definition\Type;
use GraphQL\Utils\SchemaPrinter;

/**
 * Generate GraphQL schema for Typesense documents.
 */
final class TypesenseSchemaGenerator {

  /**
   * The generated types.
   */
  protected array $types = [];

  /**
   * The generated type references as scalars.
   *
   * @var array<string, \GraphQL\Type\Definition\CustomScalarType>
   */
  protected array $scalars = [];

  /**
   * Constructs a new GraphqlTypesenseSchemaGenerator.
   */
  public function __construct() {
    $this->types['TypesenseHit'] = new InterfaceType([
      'name' => 'TypesenseHit',
      'fields' => [
        'document' => [
          'type' => Type::nonNull(fn () => $this->types['TypesenseHitDocument']),
          'description' => 'The document.',
        ],
        'text_match_info' => [
          'type' => $this->referenceType('TypesenseHitTextMatchInfo'),
          'description' => 'Details about hot the score was calculated. For internal/debugging use only.',
        ],
        'vector_distance' => [
          'type' => Type::float(),
          'description' => 'The vector distance if a vector query was performed.',
        ],
        'highlights' => [
          'type' => $this->referenceType('TypesenseHighlights'),
          'description' => 'Highlights for different fields in the search result.',
        ],
      ],
    ]);

    $this->types['TypesenseHitDocument'] = new InterfaceType([
      'name' => 'TypesenseHitDocument',
      'fields' => [
        // This field exists in all collections, because it's required by
        // Typesense. search_api_typesense makes sure that it is indexed.
        'id' => [
          'type' => Type::nonNull(Type::string()),
          'description' => 'The unique identifier of the document (not the entity!) across all languages and indices.',
        ],
      ],
    ]);

    $this->types['TypesenseSearchResults'] = new InterfaceType([
      'name' => 'TypesenseSearchResults',
      'fields' => [
        'total' => [
          'type' => Type::nonNull(Type::int()),
          'description' => 'Total number of matching documents.',
        ],
        'hits' => [
          'type' => Type::nonNull(Type::listOf(Type::nonNull($this->referenceType('TypesenseHit')))),
          'description' => 'The matching documents.',
        ],
        'facets' => [
          'type' => Type::nonNull(Type::listOf(Type::nonNull($this->referenceType('TypesenseFacet')))),
          'description' => 'The available facets.',
        ],
        'queryDebug' => [
          'type' => $this->referenceType('MapData'),
          'description' => 'Debug Query.',
        ],
      ],
    ]);
  }

  /**
   * Generate a document GraphQL type for the given index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index.
   */
  public function addIndexItemType(IndexInterface $index): void {
    $backend = $index->getServerInstance()->getBackend();
    assert($backend instanceof SearchApiTypesenseBackend);
    $schema = $backend->getSchemaForIndex($index);

    // The field configuration of the Typesense schema.
    $schemaFields = $schema->getFields();

    // The fields on the index itself.
    $indexFields = $index->getFields(TRUE);

    $graphqlFields = [];

    foreach ($indexFields as $indexField) {
      $id = $indexField->getFieldIdentifier();
      $schemaField = $schemaFields[$id] ?? NULL;
      if (!$schemaField) {
        continue;
      }

      $fieldType = $indexField->getType();
      $graphqlType = $this->getGraphqlFieldType($fieldType);

      if (!$graphqlType) {
        continue;
      }

      $optional = $schemaField['optional'] ?? FALSE;

      // Because Typesense is "strongly typed" regarding its schema, it
      // makes sure that the fields are actually indexed. We can therefore
      // also make the GraphQL field non-nullable.
      $wrappedGraphqlType = $optional
        ? $graphqlType
        : Type::nonNull($graphqlType);

      $fieldDescription = (string) $indexField->getDescription();
      $label = (string) $indexField->getLabel();

      $graphqlFields[$id] = [
        'description' => implode(
          "\n",
          array_filter([$label, $fieldDescription, 'Data Type: ' . $fieldType])
        ),
        'type' => $wrappedGraphqlType,
      ];
    }

    $indexId = $index->id();
    $indexLabel = $index->label();
    $alias = \Drupal::service('typesense_graphql.collection_manager')->getAliasForIndexId($indexId);
    $suffix = EntitySchemaHelper::toPascalCase($alias);

    // Generate a type for the document.
    $documentTypeName = EntitySchemaHelper::toPascalCase(['TypesenseHitDocument_', $suffix]);
    $this->types[$documentTypeName] = new ObjectType([
      'name' => $documentTypeName,
      'fields' => $graphqlFields,
      'description' => "Document of the '$indexId' ($indexLabel) Typesense index.",
      'interfaces' => [fn () => $this->types['TypesenseHitDocument']],
    ]);

    // Generate a type for the hit.
    $hitTypeName = EntitySchemaHelper::toPascalCase(['TypesenseHit_', $suffix]);
    $this->types[$hitTypeName] = new ObjectType([
      'name' => $hitTypeName,
      'fields' => [
        'document' => Type::nonNull(fn () => $this->types[$documentTypeName]),
      ],
      'interfaces' => [fn () => $this->types['TypesenseHit']],
    ]);

    // Generate a type for the search results.
    $this->addSearchResultsType($alias, $hitTypeName);
  }

  /**
   * Generate a TypesenseSearchResults type for the given collection alias.
   *
   * @param string $alias
   *   The collection alias.
   * @param string $hitTypeName
   *   The name of the hit type for this collection.
   */
  protected function addSearchResultsType(string $alias, string $hitTypeName): void {
    $suffix = EntitySchemaHelper::toPascalCase($alias);
    $searchResultsTypeName = 'TypesenseSearchResults' . $suffix;

    $this->types[$searchResultsTypeName] = new ObjectType([
      'name' => $searchResultsTypeName,
      'fields' => [
        'total' => [
          'type' => Type::nonNull(Type::int()),
          'description' => 'Total number of matching documents.',
        ],
        'hits' => [
          'type' => Type::nonNull(Type::listOf(Type::nonNull(fn () => $this->types[$hitTypeName]))),
          'description' => 'The matching documents. Only available if mode is RESULTS.',
        ],
        'facets' => [
          'type' => Type::nonNull(Type::listOf(Type::nonNull($this->referenceType('TypesenseFacet')))),
          'description' => 'The available facets. Only available if mode is RESULTS.',
        ],
        'queryDebug' => [
          'type' => $this->referenceType('MapData'),
          'description' => 'Debug Query.',
        ],
      ],
      'interfaces' => [fn () => $this->types['TypesenseSearchResults']],
      'description' => "Search results for the '$alias' Typesense collection.",
    ]);
  }

  /**
   * Get the GraphQL field type for a search_api_index field type.
   *
   * @param string $type
   *   The field type.
   *
   * @return \GraphQL\Type\Definition\Type|null
   *   The GraphQL type or NULL if not mapped.
   */
  protected function getGraphqlFieldType(string $type): Type|null {
    switch ($type) {
      case 'typesense_string':
        return Type::string();

      case 'typesense_string[]':
        return Type::listOf(Type::nonNull(Type::string()));

      case 'typesense_entity_reference_object_taxonomy_term[]':
      case 'typesense_entity_reference_object_node[]':
        return Type::listOf(Type::nonNull($this->referenceType('TypesenseEntityReference')));

      case 'typesense_key_value[]':
        return Type::listOf(Type::nonNull($this->referenceType('TypesenseKeyValue')));

      case 'typesense_rokka_image':
        return $this->referenceType('TypesenseRokkaImage');

      case 'typesense_date_time':
        return $this->referenceType('TypesenseDateTime');

      case 'typesense_float':
        return Type::float();

      case 'typesense_float[]':
        return Type::listOf(Type::nonNull(Type::float()));

      case 'typesense_geopoint':
        return $this->referenceType('TypesenseGeopoint');

      case 'typesense_bool':
        return Type::boolean();

      case 'typesense_bool[]':
        return Type::listOf(Type::nonNull(Type::boolean()));

      case 'typesense_int32':
      case 'typesense_int64':
        return Type::int();

      case 'typesense_int32[]':
      case 'typesense_int64[]':
        return Type::listOf(Type::nonNull(Type::int()));
    }

    return NULL;
  }

  /**
   * Reference a type that will eventually exist in the final schema.
   *
   * Usually, GraphQL requires all referenced types (in e.g. fields) to
   * exist when printing the schema. However, this class does not print
   * the entire schema, but just the types that it generates. in order for
   * these types to reference other types (either generated by
   * graphql_core_schema or in our own schema extensions), we instead reference
   * them as custom scalars. That way we don't actually generate these scalars
   * in the final output, we just reference them.
   *
   * Once the final schema is built, their reference will be valid, because the
   * type will exist at that point.
   *
   * @param string $name
   *   The name of the type.
   *
   * @return \GraphQL\Type\Definition\CustomScalarType
   *   The type reference as a custom scalar.
   */
  protected function referenceType(string $name): CustomScalarType {
    if (!empty($this->scalars[$name])) {
      return $this->scalars[$name];
    }

    $scalar = new CustomScalarType(['name' => $name]);
    $this->scalars[$name] = $scalar;
    return $scalar;
  }

  /**
   * Get the generated schema.
   *
   * Note that we don't generate an entire schema, but just print each defined
   * type individually and concatenate it. This allows us to reference types
   * that are not defined by this class, but will exist in the final schema.
   *
   * @return string
   *   The schema.
   */
  public function getGeneratedSchema(): string {
    return implode("\n", array_map(static function (Type $type) {
      return SchemaPrinter::printType($type);
    }, $this->types));
  }

}
