<?php

declare(strict_types=1);

namespace Drupal\typesense_integration\Plugin\search_api\processor;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Routing\RouteMatch;
use Drupal\node\NodeInterface;
use Drupal\search_api\Datasource\DatasourceInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\search_api\Processor\ProcessorProperty;

use Symfony\Component\Routing\Route;

/**
 * Indexes the breadcrumb for a node.
 *
 * @SearchApiProcessor(
 *   id = "typesense_breadcrumb_text",
 *   label = @Translation("Breadcrumbs"),
 *   description = @Translation("The breadcrumb for the content."),
 *   stages = {
 *     "add_properties" = 0,
 *   },
 *   locked = true,
 *   hidden = true,
 * )
 */
class Breadcrumbs extends ProcessorPluginBase {

  /**
   * {@inheritdoc}
   */
  protected function getValuesToIndex(FieldableEntityInterface $entity): array|null {
    $crumbs = [];
    if ($entity instanceof NodeInterface) {
      $breadcrumbBuilder = \Drupal::service('breadcrumb');

      $url = $entity->toUrl();
      $route = new Route('/node/{node}');
      $routeMatch = new RouteMatch($url->getRouteName(), $route, ['node' => $entity], $url->getRouteParameters());

      $links = $breadcrumbBuilder->build($routeMatch)->getLinks();

      foreach ($links as $link) {
        $crumbs[] = (string) $link->getText();
      }

    }
    return $crumbs;
  }

  /**
   * Get the property data type.
   *
   * @return string
   *   The property data type.
   */
  protected function getPropertyDataType(): string {
    return 'string';
  }

  /**
   * {@inheritdoc}
   */
  public function getPropertyDefinitions(?DatasourceInterface $datasource = NULL): array {
    $properties = [];

    if (!$datasource) {
      $pluginDefinition = $this->getPluginDefinition();
      $definition = [
        'label' => $pluginDefinition['label'],
        'description' => $pluginDefinition['description'],
        'type' => $this->getPropertyDataType(),
        'is_list' => TRUE,
        'processor_id' => $this->getPluginId(),
      ];
      $properties[$this->getPluginId()] = new ProcessorProperty($definition);
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function addFieldValues(ItemInterface $item): void {
    $fields = array_merge(
      $this->getFieldsHelper()->filterForPropertyPath($item->getFields(FALSE), $item->getDatasourceId(), $this->getPluginId()),
      $this->getFieldsHelper()->filterForPropertyPath($item->getFields(FALSE), NULL, $this->getPluginId()),
    );

    /** @var \Drupal\search_api\Item\FieldInterface|null $field */
    $field = reset($fields);

    if (!$field) {
      return;
    }

    $object = $item->getOriginalObject();
    $entity = $object->getEntity();

    if (!$entity instanceof FieldableEntityInterface) {
      return;
    }

    $values = $this->getValuesToIndex($entity) ?? [];

    foreach ($values as $value) {
      $field->addValue($value);
    }
  }

}
