<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL\DataProducer;

use Drupal\typesense_graphql\Plugin\GraphQL\TypesenseDataProducerBase;
use Drupal\graphql\GraphQL\Execution\FieldContext;

/**
 * Produces a single document by ID.
 *
 * @DataProducer(
 *   id = "typesense_graphql_document_by_id",
 *   name = "Typesense Document by ID",
 *   description = @Translation("Produce a document by ID"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The hits")
 *   ),
 *   consumes = {
 *     "id" = @ContextDefinition("string",
 *       label = @Translation("The document ID"),
 *     ),
 *     "collection" = @ContextDefinition("string",
 *       label = @Translation("The collection name"),
 *     ),
 *   }
 * )
 */
final class TypesenseDocumentById extends TypesenseDataProducerBase {

  /**
   * Resolves a single document by ID.
   *
   * @param string $id
   *   The document ID.
   * @param string $collection_name
   *   The name of the collection.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The GraphQL field context.
   *
   * @return array|null
   *   The document.
   */
  protected function resolve(string $id, string $collection_name, FieldContext $field): ?array {
    $index = $this->collectionManager->getIndexByCollectionName($collection_name);
    $alias = $this->collectionManager->getAliasForIndexId($index->id());

    if (!$index) {
      return NULL;
    }

    $field->setContextValue(
      'typesense_graphql_collection',
      $alias
    );

    $client = $this->getClient($index, $field)->getClient();

    return $client
      ->collections[$index->id()]
      ->documents[$id]
      ->retrieve();
  }

}
