<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL\DataProducer;

use Drupal\Core\Session\AccountInterface;
use Drupal\typesense_graphql\Model\TypesenseMultiSearchResponse;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;

/**
 * Produces the search hits.
 *
 * @DataProducer(
 *   id = "typesense_graphql_query_debug",
 *   name = "Debug Typesense Query",
 *   description = @Translation("Produces the debug data of the Typesense query"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The debug data")
 *   ),
 *   consumes = {
 *     "response" = @ContextDefinition("any",
 *       label = @Translation("The Typesense multiSearch response"),
 *     ),
 *     "account" = @ContextDefinition("any",
 *       label = @Translation("The current user account"),
 *     ),
 *   }
 * )
 */
final class TypesenseQueryDebug extends DataProducerPluginBase {

  /**
   * Resolves the query debug data.
   *
   * @param \Drupal\typesense_graphql\Model\TypesenseMultiSearchResponse $response
   *   An Typesense multiSearch response.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The GraphQL field context.
   *
   * @return array|null
   *   The raw query or null.
   */
  protected function resolve(
    TypesenseMultiSearchResponse $response,
    AccountInterface $account,
    FieldContext $field,
  ): ?array {
    $query = $response->getQuery();
    if (!$query) {
      return NULL;
    }

    $field->addCacheContexts(['user.permissions']);

    if (!$account->hasPermission('debug typesense search queries')) {
      return NULL;
    }

    // When debugging, don't cache anything.
    $field->mergeCacheMaxAge(0);

    $searches = $query->toQuery()['searches'] ?? [];

    foreach ($searches as $index => $search) {
      // Remove the embeddings, because they are huge.
      if (!empty($search['vector_query'])) {
        $query = $search['vector_query'];
        $searches[$index]['vector_query'] = preg_replace(
          '/(?<=\[)[^\]]*(?=\])/', '',
          $query
        );
      }
    }

    return $searches;
  }

}
