<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL;

use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_typesense\Api\TypesenseClientInterface;
use Drupal\typesense_graphql\TypesenseCollectionManager;
use Psr\Container\ContainerInterface;

/**
 * Base class for Typesense data producers.
 */
class TypesenseDataProducerBase extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new SearchTypesense data producer.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger.
   * @param \Drupal\typesense_graphql\TypesenseCollectionManager $collectionManager
   *   The collection manager.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LoggerChannelInterface $logger,
    protected TypesenseCollectionManager $collectionManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('entity_type.manager'),
      $container->get('logger.channel.typesense_graphql'),
      $container->get('typesense_graphql.collection_manager')
    );
  }

  /**
   * Get the Typesense client and add cache dependencies.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index for which to return the client.
   * @param \Drupal\Core\Cache\RefinableCacheableDependencyInterface $metadata
   *   The cacheable dependency.
   *
   * @return \Drupal\search_api_typesense\Api\TypesenseClient
   *   The client.
   */
  protected function getClient(IndexInterface $index, RefinableCacheableDependencyInterface $metadata): TypesenseClientInterface {
    // Cache tag invalidated whenever the index is updated.
    $metadata->addCacheTags(['search_api_list:' . $index->id()]);
    $metadata->addCacheableDependency($index);

    $server = $index->getServerInstance();
    $metadata->addCacheableDependency($server);

    /** @var \Drupal\search_api_typesense\Plugin\search_api\backend\SearchApiTypesenseBackend $backend */
    $backend = $server->getBackend();
    return $backend->getTypesenseClient();
  }

  /**
   * Returns the Search API index by ID.
   *
   * @param string $id
   *   The ID of the index.
   *
   * @return \Drupal\search_api\IndexInterface|null
   *   The index, or NULL if not found.
   */
  protected function getIndex(string $id): ?IndexInterface {
    $index = $this->entityTypeManager->getStorage('search_api_index')->load($id);
    if ($index instanceof IndexInterface) {
      return $index;
    }
    return NULL;
  }

}
