<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL\SchemaExtension;

use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\graphql_core_schema\EntitySchemaHelper;
use Drupal\typesense_graphql\EnumDefinitionsTrait;
use Drupal\typesense_graphql\Enum\TypesenseFacetType;
use Drupal\typesense_graphql\Enum\TypesenseSearchMode;
use Drupal\typesense_graphql\Enum\TypesenseSortDirection;
use Drupal\graphql\GraphQL\Execution\ResolveContext;
use Drupal\graphql\GraphQL\ResolverBuilder;
use Drupal\graphql\GraphQL\ResolverRegistryInterface;
use Drupal\graphql\Plugin\GraphQL\SchemaExtension\SdlSchemaExtensionPluginBase;
use Drupal\graphql_core_schema\CoreComposableConfig;
use Drupal\graphql_core_schema\SchemaBuilder\SchemaBuilderGenerator;
use Drupal\graphql_core_schema\SchemaBuilder\SchemaBuilderRegistry;
use GraphQL\Type\Definition\EnumType;
use GraphQL\Type\Definition\ResolveInfo;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\typesense_graphql\TypesenseSchemaGenerator;
use GraphQL\Type\Schema;
use GraphQL\Utils\SchemaPrinter;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\typesense_graphql\TypesenseCollectionManager;

/**
 * Extends the GraphQL schema for the Typesense search.
 *
 * @SchemaExtension(
 *   id = "typesense_graphql",
 *   name = "Typesense Search",
 *   description = "Provides fields to query the Typesense index.",
 *   schema = "core_composable"
 * )
 */
class TypesenseSearch extends SdlSchemaExtensionPluginBase implements ContainerFactoryPluginInterface, ConfigurableInterface, PluginFormInterface {

  use EnumDefinitionsTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Typesense collection manager.
   */
  protected TypesenseCollectionManager $collectionManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ModuleHandlerInterface $module_handler,
    EntityTypeManagerInterface $entityTypeManager,
    TypesenseCollectionManager $collection_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $module_handler);
    $this->entityTypeManager = $entityTypeManager;
    $this->collectionManager = $collection_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('typesense_graphql.collection_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = NestedArray::mergeDeep($this->defaultConfiguration(), $configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'enabled_collections' => [],
      'collection_aliases' => [],

    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get all Search API indices with Typesense backend using the service.
    $typesense_indices = $this->collectionManager->getAllTypesenseIndices();

    if (empty($typesense_indices)) {
      $form['no_indices'] = [
        '#type' => 'markup',
        '#markup' => $this->t('No Search API indices with Typesense backend found. Please create at least one index with a Typesense server.'),
      ];
      return $form;
    }

    $form['enabled_collections'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled collections'),
      '#description' => $this->t('Select which Typesense collections should be exposed via GraphQL.'),
      '#options' => $typesense_indices,
      '#default_value' => $this->configuration['enabled_collections'] ?? [],
      '#required' => FALSE,
    ];

    // Always show alias fields for all available collections.
    $form['collection_aliases'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Collection aliases'),
      '#description' => $this->t(
        'Configure optional aliases for collections. These aliases will be used in GraphQL queries.'
      ),
      '#tree' => TRUE,
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    foreach ($typesense_indices as $collection_id => $collection_label) {
      $form['collection_aliases'][$collection_id] = [
        '#type' => 'textfield',
        '#title' => $this->t('Alias for @collection', ['@collection' => $collection_label]),
        '#description' => $this->t('Optional alias for this collection. If empty, the collection ID will be used.'),
        '#default_value' => $this->configuration['collection_aliases'][$collection_id] ?? '',
        '#maxlength' => 255,
        '#states' => [
          'visible' => [
            ':input[name="enabled_collections[' . $collection_id . ']"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // @todo Figure out why this function is not called.
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // @todo Figure out why this function is not called.
  }

  /**
   * Get the alias for a collection.
   *
   * @param string $collection_id
   *   The collection ID.
   *
   * @return string
   *   The alias if configured, otherwise the collection ID.
   */
  public function getCollectionAlias(string $collection_id): string {
    $aliases = $this->configuration['collection_aliases'] ?? [];
    return $aliases[$collection_id] ?? $collection_id;
  }

  /**
   * Get all collection aliases.
   *
   * @return array
   *   Array of collection ID => alias mappings.
   */
  public function getCollectionAliases(): array {
    return $this->configuration['collection_aliases'] ?? [];
  }

  /**
   * Get the enabled collections.
   *
   * @return string[]
   *   Array of enabled collection IDs.
   */
  public function getEnabledCollections() {
    $configuration = $this->getConfiguration();
    return array_values(array_filter($configuration['enabled_collections'] ?? []));
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  public function getExtensionDefinition() {
    $enabled_collections = $this->getEnabledCollections();

    if (empty($enabled_collections)) {
      // If no collections are enabled, return empty schema.
      return NULL;
    }

    return parent::getExtensionDefinition();
  }

  /**
   * {@inheritdoc}
   */
  public function getBaseDefinition() {
    $generator = new SchemaBuilderGenerator();

    // Create a dynamic enum based on enabled collections.
    $enabled_collections = $this->getEnabledCollections();

    if (empty($enabled_collections)) {
      // If no collections are enabled, return empty schema.
      return NULL;
    }

    $typesenseGenerator = new TypesenseSchemaGenerator();

    // Get enabled indices using the service.
    $enabled_indices = $this->collectionManager->getEnabledIndices();

    foreach ($enabled_indices as $index) {
      $typesenseGenerator->addIndexItemType($index);
    }

    return implode("\n\n", [
      $this->buildTypesenseCollectionEnum($enabled_collections),
      $this->buildTypesenseFacetIdEnum($enabled_collections),
      $this->buildTypesenseFacetTypeEnum(),
      $this->buildTypesenseSearchModeEnum(),
      $this->buildTypesenseSortByEnum($enabled_collections),
      $this->buildTypesenseSortDirectionEnum(),
      $typesenseGenerator->getGeneratedSchema(),
      $generator->getGeneratedSchema(new SchemaBuilderRegistry(), CoreComposableConfig::fromConfiguration([]), []),
      parent::getBaseDefinition(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function registerResolvers(ResolverRegistryInterface $registry): void {
    $builder = new ResolverBuilder();

    $this->addDocumentTypeResolvers($registry);
    $this->addSearchQueryResolvers($registry, $builder);

    $registry->addFieldResolver(
      'TypesenseSearchResults',
      'total',
      $builder
        ->produce('typesense_graphql_total')
        ->map('response', $builder->fromParent())
    );

    $registry->addFieldResolver(
      'TypesenseSearchResults',
      'queryDebug',
      $builder
        ->produce('typesense_graphql_query_debug')
        ->map('response', $builder->fromParent())
        ->map('account', $builder->produce('current_user'))
    );

    $registry->addFieldResolver(
      'TypesenseSearchResults',
      'hits',
      $builder
        ->produce('typesense_graphql_hits')
        ->map('response', $builder->fromParent())
    );

    $registry->addFieldResolver(
      'TypesenseSearchResults',
      'facets',
      $builder
        ->produce('typesense_graphql_facets')
        ->map('response', $builder->fromParent())
    );

    $registry->addFieldResolver(
      'TypesenseDateTime',
      'value',
      $builder->fromParent()
    );

    $registry->addFieldResolver(
      'TypesenseDateTime',
      'parsed',
      $builder->compose(
        $builder
          ->produce('drupal_date_time')
          ->map('value', $builder->fromParent())
          ->map('format', $builder->fromValue('U'))
      )
    );

    $registry->addFieldResolver(
      'TypesenseHit',
      'highlights',
      $builder
        ->produce('typesense_graphql_highlights')
        ->map('hit', $builder->fromParent())
    );

    $registry->addFieldResolver(
      'TypesenseHighlights',
      'getHighlight',
      $builder
        ->produce('typesense_graphql_highlight_get_highlight')
        ->map('highlights', $builder->fromParent())
        ->map('field', $builder->fromArgument('field'))
    );
  }

  /**
   * Add type resolvers for Typesense documents.
   *
   * @param \Drupal\graphql\GraphQL\ResolverRegistryInterface $registry
   *   The resolver registry.
   */
  private function addDocumentTypeResolvers(ResolverRegistryInterface $registry): void {
    $registry->addTypeResolver(
      'TypesenseHit',
      static function ($value, ResolveContext $context, ResolveInfo $info) {
        $collection = $context->getContextValue($info, 'typesense_graphql_collection');
        $suffix = EntitySchemaHelper::toPascalCase($collection);
        return 'TypesenseHit' . $suffix;
      }
    );

    $registry->addTypeResolver(
      'TypesenseHitDocument',
      static function ($value, ResolveContext $context, ResolveInfo $info) {
        $collection = $context->getContextValue($info, 'typesense_graphql_collection');
        $suffix = EntitySchemaHelper::toPascalCase($collection);
        return 'TypesenseHitDocument' . $suffix;
      }
    );

    $registry->addTypeResolver(
      'TypesenseSearchResults',
      static function ($value, ResolveContext $context, ResolveInfo $info) {
        $collection = $context->getContextValue($info, 'typesense_graphql_collection');
        $suffix = EntitySchemaHelper::toPascalCase($collection);
        return 'TypesenseSearchResults' . $suffix;
      }
    );
  }

  /**
   * Add field resolvers for search queries.
   *
   * @param \Drupal\graphql\GraphQL\ResolverRegistryInterface $registry
   *   The resolver registry.
   * @param \Drupal\graphql\GraphQL\ResolverBuilder $builder
   *   The resolver builder.
   */
  private function addSearchQueryResolvers(ResolverRegistryInterface $registry, ResolverBuilder $builder): void {
    // Resolves which fields on the document are included.
    $resolveIncludeFields = $builder->callback(static function ($parent, $args, ResolveContext $context, ResolveInfo $info) {
      // This will return the names of the fields that the user has requested
      // on the document. Because the fields in the GraphQL schema correspond
      // to a field with the same name in the Typesense collection, we can
      // determine which fields we want to include when making a Typesense
      // query. Basically, we can prevent loading data from Typesense that is
      // not actually needed.
      $selection = $info->getFieldSelection(2)['hits']['document'] ?? [];
      $fields = array_keys($selection);

      // When no fields are selected to be included, fall back to just
      // including the ID. If we didn't do that, we would actually be loading
      // all fields on the document (the default behaviour), including very
      // large embedding fields.
      if (!in_array('id', $fields)) {
        $fields[] = 'id';
      }

      return $fields;
    });

    // Resolves which fields on the document should be highlighted.
    $resolveHighlightedFields = $builder->callback(static function ($parent, $args, ResolveContext $context, ResolveInfo $info) {
      $astNode = $info->fieldNodes[0];
      return array_unique(self::extractHighlightedFields($astNode));
    });

    // Resolves which fields on the document should be fully highlighted (without snippeting).
    $resolveHighlightFullFields = $builder->callback(static function ($parent, $args, ResolveContext $context, ResolveInfo $info) {
      $astNode = $info->fieldNodes[0];
      return array_unique(self::extractFullHighlightFields($astNode));
    });

    // Resolves whether hits are requested in the GraphQL query.
    // If hits are not requested, we can optimize by setting per_page to 0.
    $resolveHitsRequested = $builder->callback(static function ($parent, $args, ResolveContext $context, ResolveInfo $info) {
      $selection = $info->getFieldSelection(2);
      // Check if 'hits' field is requested in the selection.
      return isset($selection['hits']);
    });

    $registry->addFieldResolver(
      'Query',
      'searchTypesense',
      $builder->compose(
        $builder
          ->produce('typesense_graphql_search_typesense')
          ->map('language', $builder->produce('current_language'))
          ->map('includeFields', $resolveIncludeFields)
          ->map('highlightFields', $resolveHighlightedFields)
          ->map('highlightFullFields', $resolveHighlightFullFields)
          ->map('fulltextFields', $builder->fromArgument('fulltextFields'))
          ->map('collection', $builder->fromArgument('collection'))
          ->map('perPage', $builder->fromArgument('perPage'))
          ->map('page', $builder->fromArgument('page'))
          ->map('facetFields', $builder->fromArgument('facetFields'))
          ->map('selectedFacets', $builder->fromArgument('selectedFacets'))
          ->map('text', $builder->fromArgument('text'))
          ->map('hitsRequested', $resolveHitsRequested)
          ->map('searchMode', $builder->fromArgument('searchMode'))
          ->map('alpha', $builder->fromArgument('alpha'))
          ->map('distanceThreshold', $builder->fromArgument('distanceThreshold'))
          ->map('sortBy', $builder->fromArgument('sortBy')),
        $builder
          ->produce('typesense_graphql_perform_multi_search')
          ->map('query', $builder->fromParent())
      )
    );

    $registry->addFieldResolver(
      'Query',
      'typesenseDocumentById',
      $builder
        ->produce('typesense_graphql_document_by_id')
        ->map('id', $builder->fromArgument('id'))
        ->map('collection', $builder->fromArgument('collection'))
    );
  }

  /**
   * Builds the Typesense collection enum.
   *
   * @param array $collections
   *   The collections to build the enum for.
   *
   * @return string
   *   The enum schema as a string.
   */
  private function buildTypesenseCollectionEnum($collections) {
    $values = [];
    $indices = $this->entityTypeManager->getStorage('search_api_index')->loadMultiple($collections);
    $aliases = $this->getCollectionAliases();

    foreach ($indices as $index) {
      $collection_id = $index->id();
      $alias = $aliases[$collection_id] ?? NULL;

      // Use alias if available, otherwise use the index id.
      $display_name = $alias ?: $index->id();
      $key = strtoupper(str_replace('-', '_', $display_name));

      // Ensure valid enum name (cannot start with a number)
      if (is_numeric($key[0])) {
        $key = 'COLLECTION_' . $key;
      }

      $values[$key] = [
        'value' => $key,
        'description' => 'Typesense collection: ' . $display_name . ($alias ? " (alias for {$index->label()})" : ''),
      ];
    }

    $schema = new Schema([
      'types' => [
        new EnumType([
          'name' => 'TypesenseCollection',
          'values' => $values,
        ]),
      ],
    ]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Builds the Typesense facet ID enum.
   *
   * @param array $collections
   *   The collections to build the enum for.
   *
   * @return string
   *   The enum schema as a string.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function buildTypesenseFacetIdEnum($collections) {
    $values = [];
    $enabled_indices = $this->entityTypeManager->getStorage('search_api_index')->loadMultiple($collections);
    $aliases = $this->getCollectionAliases();

    // Mapping ENUM GraphQL
    // FIELDNAME -> Index, Fieldname, Type (from Index)
    // Caps -> lowercase

    /** @var \Drupal\search_api_typesense\Entity\TypesenseSchemaInterface $index */
    foreach ($enabled_indices as $index) {
      $index_id = $index->id();
      $index_label = $index->label();
      // Get alias for this collection, or use index_id if no alias is set.
      $alias = $aliases[$index_id] ?? $index_id;
      $schema_config = $index->getServerInstance()->getBackend()->getSchema($index->id())?->getSchema();

      if (!$schema_config['fields']) {
        continue;
      }

      $fields = $schema_config['fields'];

      // Get index fields to check parent field types.
      $indexFields = $index->getFields(TRUE);

      foreach ($fields as $field_config) {
        // Check if this field has facet enabled.
        if (isset($field_config['facet']) && $field_config['facet'] === TRUE) {
          $field_name = $field_config['name'] ?? '';

          // Skip parent fields of type typesense_entity_reference_object_taxonomy_term[]
          // or typesense_entity_reference_object_node[] to avoid confusion.
          // Only expose the .id subfields as facets.
          if (isset($indexFields[$field_name])) {
            $fieldType = $indexFields[$field_name]->getType();
            if ($fieldType === 'typesense_entity_reference_object_taxonomy_term[]' ||
                $fieldType === 'typesense_entity_reference_object_node[]') {
              continue;
            }
          }

          // Check if this is a subfield of a taxonomy term or node reference object.
          $hint = '';
          $enumFieldName = $field_name;
          if (strpos($field_name, '.id') !== FALSE) {
            // Extract parent field name (e.g., "field_search_terms" from "field_search_terms.id").
            $parentFieldName = substr($field_name, 0, strpos($field_name, '.id'));
            if (isset($indexFields[$parentFieldName])) {
              $parentFieldType = $indexFields[$parentFieldName]->getType();
              if ($parentFieldType === 'typesense_entity_reference_object_taxonomy_term[]') {
                $hint = ' (Taxonomy Term Entity Reference Object)';
              }
              elseif ($parentFieldType === 'typesense_entity_reference_object_node[]') {
                $hint = ' (Node Entity Reference Object)';
              }
              elseif ($parentFieldType === 'typesense_key_value[]') {
                // For KeyValue facets, use the parent field name without .id suffix.
                $enumFieldName = $parentFieldName;
                $hint = ' (Key Value)';
              }
            }
          }

          // Create enum key from the field name (may be modified for KeyValue facets).
          $key = strtoupper(str_replace(['-'], '_', $enumFieldName));
          $key = str_replace(['.'], '___', $key);
          // Ensure valid enum name (cannot start with a number)
          if (is_numeric($key[0])) {
            $key = 'FACET_' . $key;
          }

          // Create a unique key by prefixing with alias (or index_id if no alias) to avoid conflicts.
          // This must match the prefix used in mapFacetEnumsToFieldNames().
          $index_key = strtoupper(str_replace('-', '_', $alias));
          $unique_key = $index_key . '_' . $key;

          // Add the field entry.
          $values[$unique_key] = [
            'value' => $unique_key,
            'description' => "Facet $field_name from collection $index_id$hint",
          ];
        }
      }
    }

    if (empty($values)) {
      // Return empty enum if no facets found.
      return '';
    }

    $schema = new Schema([
      'types' => [
        new EnumType([
          'name' => 'TypesenseFacetId',
          'values' => $values,
        ]),
      ],
    ]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Builds the Typesense facet type enum.
   *
   * @return string
   *   The enum schema as a string.
   */
  private function buildTypesenseFacetTypeEnum() {
    $values = [];

    foreach (TypesenseFacetType::cases() as $case) {
      $key = $case->toGraphQlEnumValue();
      $values[$key] = [
        'value' => $key,
        'description' => ucfirst(str_replace('_', ' ', $case->value)),
      ];
    }

    $schema = new Schema([
      'types' => [
        new EnumType([
          'name' => 'TypesenseFacetType',
          'values' => $values,
        ]),
      ],
    ]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Builds the Typesense search mode enum.
   *
   * @return string
   *   The enum schema as a string.
   */
  private function buildTypesenseSearchModeEnum() {
    $values = [];

    foreach (TypesenseSearchMode::cases() as $case) {
      $key = $case->toGraphQlEnumValue();
      $descriptions = [
        'KEYWORD_SEARCH' => 'Keyword search mode (default). Uses traditional full-text search.',
        'SEMANTIC_SEARCH' => 'Semantic search mode. Uses vector embeddings for semantic similarity search. Only available when embeddings are enabled on the collection.',
        'HYBRID_SEARCH' => 'Hybrid search mode. Combines keyword and semantic search. Only available when embeddings are enabled on the collection.',
      ];
      $values[$key] = [
        'value' => $key,
        'description' => $descriptions[$key] ?? ucfirst(str_replace('_', ' ', $case->value)),
      ];
    }

    $schema = new Schema([
      'types' => [
        new EnumType([
          'name' => 'TypesenseSearchMode',
          'values' => $values,
        ]),
      ],
    ]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Builds the Typesense sort by enum.
   *
   * @param array $collections
   *   The collections to build the enum for.
   *
   * @return string
   *   The enum schema as a string.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function buildTypesenseSortByEnum($collections) {
    $values = [];
    $enabled_indices = $this->entityTypeManager->getStorage('search_api_index')->loadMultiple($collections);
    $aliases = $this->getCollectionAliases();

    /** @var \Drupal\search_api_typesense\Entity\TypesenseSchemaInterface $index */
    foreach ($enabled_indices as $index) {
      $index_id = $index->id();
      // Get alias for this collection, or use index_id if no alias is set.
      $alias = $aliases[$index_id] ?? $index_id;
      $schema_config = $index->getServerInstance()->getBackend()->getSchema($index->id())?->getSchema();

      if (!$schema_config || !isset($schema_config['fields'])) {
        continue;
      }

      $fields = $schema_config['fields'];

      foreach ($fields as $field_config) {
        // Check if this field has sort enabled.
        if (isset($field_config['sort']) && $field_config['sort'] === TRUE) {
          $field_name = $field_config['name'] ?? '';
          if (!$field_name) {
            continue;
          }

          // Create enum key from the field name.
          $key = strtoupper(str_replace(['-'], '_', $field_name));
          $key = str_replace(['.'], '___', $key);
          // Ensure valid enum name (cannot start with a number)
          if (is_numeric($key[0])) {
            $key = 'SORT_' . $key;
          }

          // Create a unique key by prefixing with alias (or index_id if no alias) to avoid conflicts.
          $index_key = strtoupper(str_replace('-', '_', $alias));
          $unique_key = $index_key . '_' . $key;

          // Add the field entry.
          $values[$unique_key] = [
            'value' => $unique_key,
            'description' => "Sortable field $field_name from collection $index_id",
          ];
        }
      }
    }

    if (empty($values)) {
      // Return empty enum if no sortable fields found.
      return '';
    }

    $schema = new Schema([
      'types' => [
        new EnumType([
          'name' => 'TypesenseSortBy',
          'values' => $values,
        ]),
      ],
    ]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Builds the Typesense sort direction enum.
   *
   * @return string
   *   The enum schema as a string.
   */
  private function buildTypesenseSortDirectionEnum() {
    $values = [];

    foreach (TypesenseSortDirection::cases() as $case) {
      $key = $case->toGraphQlEnumValue();
      $descriptions = [
        'ASC' => 'Sort in ascending order.',
        'DESC' => 'Sort in descending order.',
      ];
      $values[$key] = [
        'value' => $key,
        'description' => $descriptions[$key] ?? ucfirst($case->value),
      ];
    }

    $schema = new Schema([
      'types' => [
        new EnumType([
          'name' => 'TypesenseSortDirection',
          'values' => $values,
        ]),
      ],
    ]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Recursively extracts highlighted field names from GraphQL AST nodes.
   *
   * @param object $node
   *   The GraphQL AST node to traverse.
   *
   * @return array
   *   Array of field names that should be highlighted.
   */
  private static function extractHighlightedFields($node): array {
    $highlightedFields = [];

    // If this is a getHighlight field with arguments, extract the field argument
    if (isset($node->name) && $node->name->value === 'getHighlight' && isset($node->arguments)) {
      foreach ($node->arguments as $argument) {
        if ($argument->name->value === 'field' && $argument->value->kind === 'StringValue') {
          $highlightedFields[] = $argument->value->value;
        }
      }
    }

    // Recursively traverse child selections
    if (isset($node->selectionSet) && isset($node->selectionSet->selections)) {
      foreach ($node->selectionSet->selections as $selection) {
        $highlightedFields = array_merge($highlightedFields, self::extractHighlightedFields($selection));
      }
    }

    return $highlightedFields;
  }

  /**
   * Recursively extracts full highlight field names from GraphQL AST nodes.
   *
   * These are fields that should be highlighted fully without snippeting.
   *
   * @param object $node
   *   The GraphQL AST node to traverse.
   *
   * @return array
   *   Array of field names that should be fully highlighted.
   */
  private static function extractFullHighlightFields($node): array {
    $fullHighlightFields = [];

    // If this is a getHighlight field with arguments, check if fullField is true
    if (isset($node->name) && $node->name->value === 'getHighlight' && isset($node->arguments)) {
      $fieldName = NULL;
      $fullField = FALSE;

      foreach ($node->arguments as $argument) {
        if ($argument->name->value === 'field' && $argument->value->kind === 'StringValue') {
          $fieldName = $argument->value->value;
        }
        elseif ($argument->name->value === 'fullField' && $argument->value->kind === 'BooleanValue') {
          $fullField = $argument->value->value;
        }
      }

      // If fullField is true, add this field to the list
      if ($fullField && $fieldName) {
        $fullHighlightFields[] = $fieldName;
      }
    }

    // Recursively traverse child selections
    if (isset($node->selectionSet) && isset($node->selectionSet->selections)) {
      foreach ($node->selectionSet->selections as $selection) {
        $fullHighlightFields = array_merge($fullHighlightFields, self::extractFullHighlightFields($selection));
      }
    }

    return $fullHighlightFields;
  }

}
