<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL\DataProducer\Query;

use Drupal\typesense_graphql\Model\TypesenseQuery;
use Drupal\typesense_graphql\Plugin\GraphQL\TypesenseDataProducerBase;
use Drupal\typesense_graphql\Enum\TypesenseSearchMode;
use Drupal\graphql\GraphQL\Execution\FieldContext;

/**
 * Provides a typesense search data producer.
 *
 * @DataProducer(
 *   id = "typesense_graphql_search_typesense",
 *   name = "Search Typesense",
 *   description = @Translation("Search the Typesense index."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The search results"),
 *   ),
 *   consumes = {
 *     "language" = @ContextDefinition("string",
 *       label = @Translation("The language"),
 *     ),
 *     "includeFields" = @ContextDefinition("string",
 *       label = @Translation("The fields to include."),
 *       multiple = TRUE,
 *     ),
 *     "highlightFields" = @ContextDefinition("string",
 *       label = @Translation("The fields to highlight."),
 *       multiple = TRUE,
 *       required = FALSE
 *     ),
 *     "highlightFullFields" = @ContextDefinition("string",
 *       label = @Translation("The fields to highlight fully without snippeting."),
 *       multiple = TRUE,
 *       required = FALSE
 *     ),
 *     "fulltextFields" = @ContextDefinition("any",
 *       label = @Translation("The fulltext fields configuration"),
 *       multiple = TRUE,
 *       required = TRUE,
 *     ),
 *     "perPage" = @ContextDefinition("integer",
 *       label = @Translation("Results per page"),
 *       required = FALSE,
 *       default_value = 30
 *     ),
 *     "page" = @ContextDefinition("integer",
 *       label = @Translation("Page"),
 *       required = FALSE,
 *       default_value = 0
 *     ),
 *     "collection" = @ContextDefinition("string",
 *       label = @Translation("Collection"),
 *     ),
 *     "text" = @ContextDefinition("string",
 *       label = @Translation("The search text"),
 *       required = FALSE
 *     ),
 *     "facetFields" = @ContextDefinition("string",
 *       label = @Translation("The facet fields to return"),
 *       multiple = TRUE,
 *       required = FALSE
 *     ),
 *     "selectedFacets" = @ContextDefinition("any",
 *       label = @Translation("The selected facets for filtering"),
 *       required = FALSE
 *     ),
 *     "hitsRequested" = @ContextDefinition("boolean",
 *       label = @Translation("Whether hits are requested in the GraphQL query"),
 *       required = FALSE,
 *       default_value = TRUE
 *     ),
 *     "searchMode" = @ContextDefinition("string",
 *       label = @Translation("The search mode"),
 *       required = FALSE
 *     ),
 *     "alpha" = @ContextDefinition("float",
 *       label = @Translation("The alpha parameter for hybrid search"),
 *       required = FALSE
 *     ),
 *     "distanceThreshold" = @ContextDefinition("float",
 *       label = @Translation("The distance threshold for vector search"),
 *       required = FALSE
 *     ),
 *     "sortBy" = @ContextDefinition("any",
 *       label = @Translation("How to sort the search results"),
 *       required = FALSE
 *     ),
 *   }
 * )
 */
final class SearchTypesense extends TypesenseDataProducerBase {

  /**
   * Resolves the search results.
   *
   * @param string $language
   *   The language.
   * @param string[] $includeFields
   *   The fields to include.
   * @param string[] $highlightFields
   *   The fields to highlight.
   * @param string[] $highlightFullFields
   *   The fields to highlight fully without snippeting.
   * @param array $fulltextFields
   *   The TypesenseFulltextFieldInput input type.
   * @param int $perPage
   *   How many results to load per page.
   * @param int $page
   *   The current page.
   * @param string $collection
   *   The collection.
   * @param string|null $text
   *   The search text.
   * @param array|null $facetFields
   *   The TypesenseFacetId enum values to return.
   * @param array|null $selectedFacets
   *   The TypesenseSelectedFacet input type for filtering.
   * @param bool $hitsRequested
   *   Whether hits are requested in the GraphQL query.
   * @param string|null $searchMode
   *   The search mode: 'KEYWORD_SEARCH', 'SEMANTIC_SEARCH', or 'HYBRID_SEARCH'.
   * @param float|null $alpha
   *   The alpha parameter for hybrid search.
   * @param float|null $distanceThreshold
   *   The distance threshold for vector search.
   * @param array|null $sortBy
   *   The TypesenseSortByInput input type for sorting.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The GraphQL field context.
   *
   * @return array|null
   *   The Typesense response or null if an error occured.
   */
  protected function resolve(
    string $language,
    array $includeFields,
    array $highlightFields,
    array $highlightFullFields,
    array $fulltextFields,
    int $perPage,
    int $page,
    string $collection,
    ?string $text,
    ?array $facetFields,
    ?array $selectedFacets,
    bool $hitsRequested,
    ?string $searchMode,
    ?float $alpha,
    ?float $distanceThreshold,
    ?array $sortBy,
    FieldContext $field,
  ): TypesenseQuery {

    /** @var \Drupal\search_api\IndexInterface $index */
    $index = $this->collectionManager->getIndexByCollectionName($collection);
    if (!$index) {
      throw new \InvalidArgumentException('Index not found for collection: ' . $collection);
    }
    $alias = $this->collectionManager->getAliasForIndexId($index->id());

    // Determine which facet fields to return.
    $facetFieldNames = $this->getFacetFieldNames($index, $alias, $facetFields);

    // Determine which facets are entity reference facets (taxonomy term or node reference).
    // These need facet_return_parent to get full object data.
    $entityReferenceFacetFields = $this->getEntityReferenceFacetFields($index, $facetFieldNames);

    $query = new TypesenseQuery($index, $facetFieldNames, $entityReferenceFacetFields);

    // If hits are not requested, set per_page to 0 for performance.
    // This prevents Typesense from fetching and returning hit documents.
    $effectivePerPage = $hitsRequested ? $perPage : 0;

    $query
      ->addIncludeFields($includeFields)
      ->addHighlightFields($highlightFields)
      ->addHighlightFullFields($highlightFullFields)
      ->setAlias($alias)
      ->setPerPage($effectivePerPage)
      ->setPage($page);

    // Check if the index has a field called 'langcode'.
    if ($index->getField('langcode')) {
      $query->addEqualsFilter('langcode', $language);
    }

    if ($text) {
      $query->setSearchText($text);

      if (!empty($fulltextFields)) {
        // Use the provided fulltext fields configuration
        foreach ($fulltextFields as $fieldConfig) {
          $fulltextField = $query->defineFulltextField($fieldConfig['name']);

          // Set weight if set.
          if (!empty($fieldConfig['weight'])) {
            $weight = (int) $fieldConfig['weight'];
            $fulltextField->weight($weight);
          }

          // Set prefix with default fallback to false
          if (!empty($fieldConfig['prefix'])) {
            $fulltextField->prefix();
          }

          if (!empty($fieldConfig['infix'])) {
            $fulltextField->infix($fieldConfig['infix']);
          }

          // Set numTypos with default fallback to 0
          $numTypos = !empty($fieldConfig['numTypos']) ? $fieldConfig['numTypos'] : 0;
          if ($numTypos > 0) {
            $fulltextField->numTypos($numTypos);
          }
        }
      }

      // If no explicit sortBy is provided, default to text match relevance.
      if (empty($sortBy)) {
        $query->setSortBy('_text_match:desc');
      }
    }

    // Handle explicit sortBy if provided.
    if (!empty($sortBy)) {
      $sortFields = [];
      foreach ($sortBy as $sortItem) {
        if (isset($sortItem['field'])) {
          $sortFieldName = $this->mapSortByEnumToFieldName($index, $alias, $sortItem['field']);
          $sortDirection = $sortItem['direction'] ?? 'DESC';
          // Convert GraphQL enum value (ASC/DESC) to lowercase (asc/desc) for Typesense.
          $sortDirection = strtolower($sortDirection);
          $sortFields[] = $sortFieldName . ':' . $sortDirection;
        }
      }
      if (!empty($sortFields)) {
        // Join multiple sort fields with commas for Typesense.
        $query->setSortBy(implode(',', $sortFields));
      }
    }

    // Set search mode (default to KEYWORD_SEARCH if not specified).
    $effectiveSearchMode = $searchMode ?: TypesenseSearchMode::KeywordSearch->toGraphQlEnumValue();
    $query->setSearchMode($effectiveSearchMode);

    // Set alpha and distanceThreshold if provided, otherwise use defaults.
    if ($alpha !== NULL) {
      $query->setAlpha($alpha);
    }
    if ($distanceThreshold !== NULL) {
      $query->setDistanceThreshold($distanceThreshold);
    }

    // Validate that semantic/hybrid search modes require embeddings.
    $semanticModes = [
      TypesenseSearchMode::SemanticSearch->toGraphQlEnumValue(),
      TypesenseSearchMode::HybridSearch->toGraphQlEnumValue(),
    ];
    if (in_array($effectiveSearchMode, $semanticModes, TRUE)) {
      /** @var \Drupal\search_api_typesense\Plugin\search_api\backend\SearchApiTypesenseBackend $backend */
      $backend = $index->getServerInstance()->getBackend();
      $schema = $backend->getSchemaForIndex($index);
      if (!$schema || !$schema->isEmbeddingEnabled()) {
        throw new \InvalidArgumentException('Semantic and hybrid search modes require embeddings to be enabled on the collection.');
      }
    }

    if (!empty($selectedFacets)) {
      // Filter out facets with null values, empty strings, empty values arrays, or arrays containing only null/empty values -
      // these should be treated as if no facet filter was provided.
      $selectedFacets = array_filter($selectedFacets, function (array $facet) {
        // If values is null or not set, filter out this facet.
        if (!isset($facet['values']) || $facet['values'] === NULL) {
          return FALSE;
        }
        // If values is an empty array, filter out this facet.
        if (empty($facet['values'])) {
          return FALSE;
        }
        // Filter out null values and empty strings from the array, then check if anything remains.
        $nonNullValues = array_filter($facet['values'], fn($v) => $v !== NULL && $v !== '');
        return !empty($nonNullValues);
      });

      if (!empty($selectedFacets)) {
        $prefix = $this->collectionManager->getAliasForIndexId($index->id());
        $indexFields = $index->getFields(TRUE);
        $facetsMapped = array_map(function (array $facet) use ($prefix, $indexFields) {
          // Remove the collection prefix and convert to lowercase.
          $fieldKey = strtolower(preg_replace('/^' . preg_quote(strtoupper($prefix), '/') . '_/', '', $facet['id']));
          $fieldKey = str_replace('___', '.', $fieldKey);

          // Check if this is a taxonomy term, node reference, or KeyValue facet parent field.
          if (isset($indexFields[$fieldKey])) {
            $parentFieldType = $indexFields[$fieldKey]->getType();
            if ($parentFieldType === 'typesense_entity_reference_object_taxonomy_term[]' ||
                $parentFieldType === 'typesense_entity_reference_object_node[]' ||
                $parentFieldType === 'typesense_key_value[]') {
              // Map to the .id subfield.
              $fieldKey = $fieldKey . '.id';
            }
          }

          return array_merge($facet, [
            'id' => $fieldKey,
          ]);
        }, $selectedFacets);
        $query->setSelectedFacets($facetsMapped);
      }
    }

    return $query;
  }

  /**
   * Gets the facet field names to use in the query.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param string $alias
   *   The collection alias.
   * @param array|null $facetFields
   *   The requested facet field enum values. If NULL or empty, no facets will be returned.
   *
   * @return string[]
   *   Array of facet field names. Empty array if no facets requested.
   */
  protected function getFacetFieldNames($index, string $alias, ?array $facetFields): array {
    // Only return facets if explicitly requested.
    if (!empty($facetFields)) {
      return $this->mapFacetEnumsToFieldNames($index, $alias, $facetFields);
    }

    // If facetFields is not provided or empty, return no facets for performance.
    return [];
  }

  /**
   * Maps GraphQL facet enum values to Typesense field names.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param string $alias
   *   The collection alias.
   * @param array $facetEnums
   *   Array of GraphQL enum values (e.g., ['CONTENT_BUNDLE', 'CONTENT_TYPE']).
   *
   * @return string[]
   *   Array of Typesense field names.
   */
  protected function mapFacetEnumsToFieldNames($index, string $alias, array $facetEnums): array {
    $fieldNames = [];
    $backend = $index->getServerInstance()->getBackend();
    // Use the same approach as TypesenseSearch.php for consistency.
    $schema_config = $index->getServerInstance()->getBackend()->getSchema($index->id());
    if (!$schema_config) {
      return $fieldNames;
    }
    $schema_array = $schema_config->getSchema();

    if (!$schema_array || !isset($schema_array['fields'])) {
      return $fieldNames;
    }

    $fields = $schema_array['fields'];
    $prefix = strtoupper(str_replace('-', '_', $alias));
    $indexFields = $index->getFields(TRUE);

    foreach ($facetEnums as $enum) {
      // Remove the collection prefix (e.g., "CONTENT_" from "CONTENT_BUNDLE").
      $fieldKey = strtolower(preg_replace('/^' . preg_quote($prefix, '/') . '_/', '', $enum));
      $fieldKey = str_replace('___', '.', $fieldKey);

      // First, check if this might be an entity reference facet or KeyValue facet parent field.
      // If the parent field exists and is of type entity_reference_object_taxonomy_term[],
      // entity_reference_object_node[], or typesense_key_value[], map it to the .id subfield.
      if (isset($indexFields[$fieldKey])) {
        $parentFieldType = $indexFields[$fieldKey]->getType();
        if ($parentFieldType === 'typesense_entity_reference_object_taxonomy_term[]' ||
            $parentFieldType === 'typesense_entity_reference_object_node[]' ||
            $parentFieldType === 'typesense_key_value[]') {
          // This is an entity reference or KeyValue facet, use the .id subfield.
          $facetFieldName = $fieldKey . '.id';
          // Verify this facet field exists in the schema.
          foreach ($fields as $field_config) {
            if (isset($field_config['facet']) && $field_config['facet'] === TRUE) {
              if (($field_config['name'] ?? '') === $facetFieldName) {
                $fieldNames[] = $facetFieldName;
                // Continue outer loop.
                continue 2;
              }
            }
          }
        }
      }

      // Otherwise, find the field directly in the schema.
      foreach ($fields as $field_config) {
        if (isset($field_config['facet']) && $field_config['facet'] === TRUE) {
          $fieldName = $field_config['name'] ?? '';
          // Normalize field name for comparison.
          $normalizedFieldName = strtolower(str_replace(['-', '.'], ['_', '___'], $fieldName));
          if ($normalizedFieldName === $fieldKey || $fieldName === $fieldKey) {
            $fieldNames[] = $fieldName;
            break;
          }
        }
      }
    }

    return $fieldNames;
  }

  /**
   * Detects all facet-enabled fields from the index schema.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   *
   * @return string[]
   *   Array of facet field names.
   */
  protected function detectFacetFieldsFromIndex($index): array {
    $fieldNames = [];
    // Use the same approach as TypesenseSearch.php for consistency.
    $schema_config = $index->getServerInstance()->getBackend()->getSchema($index->id());
    if (!$schema_config) {
      return $fieldNames;
    }
    $schema_array = $schema_config->getSchema();

    if (!$schema_array || !isset($schema_array['fields'])) {
      return $fieldNames;
    }

    $fields = $schema_array['fields'];

    foreach ($fields as $field_config) {
      if (isset($field_config['facet']) && $field_config['facet'] === TRUE) {
        $fieldName = $field_config['name'] ?? '';
        if ($fieldName) {
          $fieldNames[] = $fieldName;
        }
      }
    }

    return $fieldNames;
  }

  /**
   * Determines which facet fields require facet_return_parent to get full object data.
   *
   * These include entity reference facets (taxonomy term or node reference) and
   * KeyValue facets.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param string[] $facetFieldNames
   *   Array of facet field names.
   *
   * @return string[]
   *   Array of facet field names that require facet_return_parent.
   */
  protected function getEntityReferenceFacetFields($index, array $facetFieldNames): array {
    $facetsRequiringParent = [];
    $indexFields = $index->getFields(TRUE);

    foreach ($facetFieldNames as $facetFieldName) {
      // Check if this is an entity reference facet by checking if the parent field
      // is of type entity_reference_object_taxonomy_term[] or entity_reference_object_node[].
      if (strpos($facetFieldName, '.id') !== FALSE) {
        // Extract parent field name (e.g., "field_search_terms" from "field_search_terms.id").
        $parentFieldName = substr($facetFieldName, 0, strpos($facetFieldName, '.id'));
        if (isset($indexFields[$parentFieldName])) {
          $parentFieldType = $indexFields[$parentFieldName]->getType();
          if ($parentFieldType === 'typesense_entity_reference_object_taxonomy_term[]' ||
              $parentFieldType === 'typesense_entity_reference_object_node[]' ||
              $parentFieldType === 'typesense_key_value[]') {
            $facetsRequiringParent[] = $facetFieldName;
          }
        }
      }
    }

    return $facetsRequiringParent;
  }

  /**
   * Maps GraphQL sort by enum value to Typesense field name.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param string $alias
   *   The collection alias.
   * @param string $sortByEnum
   *   The GraphQL enum value (e.g., 'CONTENT_ENTITY_ID').
   *
   * @return string
   *   The Typesense field name.
   */
  protected function mapSortByEnumToFieldName($index, string $alias, string $sortByEnum): string {
    $backend = $index->getServerInstance()->getBackend();
    $schema_config = $index->getServerInstance()->getBackend()->getSchema($index->id());
    if (!$schema_config) {
      throw new \InvalidArgumentException('Schema not found for index: ' . $index->id());
    }
    $schema_array = $schema_config->getSchema();

    if (!$schema_array || !isset($schema_array['fields'])) {
      throw new \InvalidArgumentException('Schema fields not found for index: ' . $index->id());
    }

    $fields = $schema_array['fields'];
    $prefix = strtoupper(str_replace('-', '_', $alias));

    // Remove the collection prefix (e.g., "CONTENT_" from "CONTENT_ENTITY_ID").
    $fieldKey = strtolower(preg_replace('/^' . preg_quote($prefix, '/') . '_/', '', $sortByEnum));
    $fieldKey = str_replace('___', '.', $fieldKey);

    // Find the field in the schema.
    foreach ($fields as $field_config) {
      if (isset($field_config['sort']) && $field_config['sort'] === TRUE) {
        $fieldName = $field_config['name'] ?? '';
        // Normalize field name for comparison.
        $normalizedFieldName = strtolower(str_replace(['-', '.'], ['_', '___'], $fieldName));
        if ($normalizedFieldName === $fieldKey || $fieldName === $fieldKey) {
          return $fieldName;
        }
      }
    }

    throw new \InvalidArgumentException('Sortable field not found for enum: ' . $sortByEnum);
  }

}
