<?php

declare(strict_types=1);

namespace Drupal\typesense_integration\Plugin\search_api\processor;

use Drupal\Core\Entity\EntityInterface;
use Drupal\search_api\Processor\ProcessorPluginBase;

/**
 * Filters out entities if they are flagged not to be indexed.
 *
 * @SearchApiProcessor(
 *   id = "typesense_remove_excluded",
 *   label = @Translation("Exclude content configured not to be indexed"),
 *   description = @Translation("Filters out entities that are configured not to be indexed."),
 *   stages = {
 *     "alter_items" = 0,
 *   },
 * )
 */
class RemoveExcluded extends ProcessorPluginBase {

  /**
   * {@inheritdoc}
   */
  public function alterIndexedItems(array &$items): void {
    /** @var \Drupal\search_api\Item\ItemInterface $item */
    foreach ($items as $item_id => $item) {
      $value = $item->getOriginalObject()?->getValue();
      if ($value instanceof EntityInterface) {
        if ($this->shouldExclude($value)) {
          unset($items[$item_id]);
        }
      }
    }
  }

  /**
   * Determine if the entity should be excluded.
   *
   * @param EntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if the entity should be excluded.
   */
  private function shouldExclude(EntityInterface $entity): bool {
    if ($entity->hasField('field_exclude_from_search')) {
      $is_excluded = (bool) $entity->get('field_exclude_from_search')->value;
      return $is_excluded;
    }

    return FALSE;
  }

}
