<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\search_api\IndexInterface;

/**
 * Service to manage Typesense collections.
 */
class TypesenseCollectionManager {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new TypesenseCollectionManager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * Get all enabled collections over all GraphQL servers.
   *
   * @return array
   *   Array of index_id => alias pairs. If there is no alias, returns index_id => index_id.
   */
  public function getEnabledCollections(): array {
    $result = [];

    // Get all GraphQL server configurations
    $config = $this->configFactory->get('graphql.graphql_servers.graphql');
    $schema_configuration = $config->get('schema_configuration') ?? [];

    // Find all servers that use our extension
    foreach ($schema_configuration as $schema_config) {
      $extension_config = $schema_config['extension_typesense_graphql'] ?? [];

      if (!empty($extension_config)) {
        $enabled_collections = $extension_config['enabled_collections'] ?? [];
        $collection_aliases = $extension_config['collection_aliases'] ?? [];

        // Add all enabled collections to the result
        foreach ($enabled_collections as $index_id => $enabled) {
          if ($enabled) {
            // Use alias if available, otherwise use index_id
            $alias = isset($collection_aliases[$index_id]) && !empty($collection_aliases[$index_id])
              ? $collection_aliases[$index_id]
              : $index_id;

            $result[$index_id] = $alias;
          }
        }
      }
    }

    return $result;
  }

  /**
   * Get enabled indices.
   *
   * @return \Drupal\search_api\IndexInterface[]
   *   Array of enabled indices.
   */
  public function getEnabledIndices(): array {
    $enabled_collection_ids = $this->getEnabledCollections();
    $ids = array_keys($enabled_collection_ids);
    $index_storage = $this->entityTypeManager->getStorage('search_api_index');
    return $index_storage->loadMultiple($ids);
  }

  /**
   * Get all Search API indices that use Typesense backend.
   *
   * Returns all available Typesense indices, regardless of whether
   * they are enabled in any GraphQL server configuration.
   *
   * @return array
   *   Array of index_id => index_label pairs.
   */
  public function getAllTypesenseIndices(): array {
    $result = [];
    $index_storage = $this->entityTypeManager->getStorage('search_api_index');
    $indices = $index_storage->loadMultiple();

    foreach ($indices as $index) {
      if ($index->status() && $index->getServerInstance()?->getBackendId() === 'search_api_typesense') {
        $result[$index->id()] = $index->label();
      }
    }

    return $result;
  }

  /**
   * Get the search API index from a GraphQL collection name.
   *
   * This method performs a reverse lookup to find the Search API index
   * from a GraphQL collection name (like "CONTENT"). It searches through the
   * configured collection aliases and compares them (uppercased and with hyphens
   * replaced by underscores) with the input collection name. If no alias is found,
   * it also checks the enabled collection machine names directly.
   *
   * @param string $collectionName
   *   The GraphQL collection name (e.g., "CONTENT", "JOB_POSTINGS").
   *
   * @return \Drupal\search_api\IndexInterface|null
   *   The index if found, null otherwise.
   */
  public function getIndexByCollectionName(string $collectionName): ?IndexInterface {
    // Get all enabled collections with their aliases
    $enabled_collections = $this->getEnabledCollections();

    foreach ($enabled_collections as $collection_id => $alias) {
      // First, check if the alias matches (uppercased and normalized)
      $uppercase_alias = strtoupper(str_replace('-', '_', $alias));
      if ($uppercase_alias === $collectionName) {
        return $this->entityTypeManager->getStorage('search_api_index')->load($collection_id);
      }

      // If no alias match, check if the collection ID matches
      $uppercase_collection_id = strtoupper(str_replace('-', '_', $collection_id));
      if ($uppercase_collection_id === $collectionName) {
        return $this->entityTypeManager->getStorage('search_api_index')->load($collection_id);
      }
    }

    return NULL;
  }

  public function getAliasForIndexId(string $indexId): string {
    // Use the existing getEnabledCollections method to get all collections with aliases
    $enabled_collections = $this->getEnabledCollections();

    // Return the alias if the index is enabled, otherwise return the index ID
    return $enabled_collections[$indexId] ?? $indexId;
  }

}
