## INTRODUCTION

The Typesense GraphQL module provides comprehensive GraphQL integration for Typesense search collections in Drupal. It enables you to expose your Typesense search indexes as fully-featured GraphQL APIs with dynamic schema generation, faceted search, semantic search capabilities, and more.

The module consists of two parts:

- **Main module** (`typesense_graphql`): Core GraphQL integration with dynamic schema generation, search queries, and faceting
- **Integration submodule** (`typesense_integration`): Additional Search API processors for enhanced indexing capabilities

Key capabilities:

- Expose Typesense search collections via GraphQL with dynamic schema generation
- Full-text search with configurable field weights, typo tolerance, and search modes
- Faceted search with filtering and selective facet loading
- Semantic and hybrid search support (when embeddings are enabled)
- Search result highlighting and detailed match information
- Language-aware search and document retrieval
- Performance optimizations for large-scale search applications

### Requirements

- Drupal 10 or 11
- [search_api_typesense module](https://www.drupal.org/project/search_api_typesense)
- [GraphQL module](https://www.drupal.org/project/graphql)

### Installation

Install as you would normally install a contributed Drupal module.
See: https://www.drupal.org/node/895232 for further information.

### Configuration

1. Go to Administration > Configuration > Web services > GraphQL > Servers
2. Edit the "core_composable" server (or your GraphQL server)
3. In the "Schema Extensions" section, find "Typesense Search"
4. Click "Configure" next to the Typesense Search extension
5. Select which Typesense collections should be exposed via GraphQL
6. Save the server configuration
7. The GraphQL schema will be automatically updated to include only the enabled collections

## FEATURES

### GraphQL Queries

#### Search Query (`searchTypesense`)

Perform full-text searches across Typesense collections with support for:

- **Full-text search**: Search across multiple fields with configurable weights
- **Field configuration**: Per-field settings including:
  - Search weight
  - Prefix search (enabled/disabled)
  - Infix search (always/fallback/off)
  - Typo tolerance (numTypos)
- **Pagination**: `page` and `perPage` parameters for result pagination
- **Language-aware**: Automatically filters results by language when `langcode` field exists
- **Field selection**: Include/exclude specific fields for performance optimization
- **Search modes**:
  - `KEYWORD_SEARCH`: Traditional keyword-based search (default)
  - `SEMANTIC_SEARCH`: Vector-based semantic search (requires embeddings)
  - `HYBRID_SEARCH`: Combines keyword and semantic search (requires embeddings)
- **Hybrid search parameters**:
  - `alpha`: Controls ratio between keyword and semantic scores (0.0-1.0, default: 0.8)
  - `distanceThreshold`: Vector distance threshold (default: 0.5)
- **Highlights**: Field-level search result highlighting with snippet generation
- **Text match info**: Detailed scoring information for debugging

#### Document Query (`typesenseDocumentById`)

Retrieve individual documents by their Typesense document ID:

- Direct document access without search
- Language-aware document retrieval
- Full document data with all indexed fields

### Faceted Search

- **Dynamic facet enums**: Automatically generates `TypesenseFacetId` enum based on collection schema
- **Facet filtering**: Filter results using `selectedFacets` parameter
- **Selective facet return**: Use `facetFields` to specify which facets to return (performance optimization)
- **Nested facets**: Support for hierarchical/taxonomy term facets with parent-child relationships
- **Facet metadata**: Includes labels, counts, weights, and multilingual labels (`all_labels`)
- **Entity reference facets**: Special handling for taxonomy term and node reference facets
- **KeyValue facets**: Support for key-value pair facets (e.g., page bundles with machine name and label)

### Dynamic Schema Generation

The module automatically generates GraphQL types based on your Typesense collection schemas:

- **Collection enums**: Dynamic `TypesenseCollection` enum based on enabled collections
- **Document types**: Type-safe document types for each collection (e.g., `TypesenseHitDocumentContent`)
- **Hit types**: Search result hit types with document, highlights, and match info
- **Search result types**: Collection-specific search result types
- **Facet types**: Dynamic facet ID enums and facet term types
- **Data type support**:
  - Strings (single and arrays)
  - Integers (int32, int64, single and arrays)
  - Floats (single and arrays)
  - Booleans (single and arrays)
  - Date/time (`TypesenseDateTime`)
  - Geopoints (`TypesenseGeopoint`)
  - Entity references (`TypesenseEntityReference`) for taxonomy terms and nodes
  - KeyValue objects (`TypesenseKeyValue`) for key-value pairs with id and label
  - Rokka images (`TypesenseRokkaImage`)

### GraphQL Server Integration

- Configuration stored within GraphQL server settings
- No separate configuration forms or files needed
- Settings persist with the GraphQL server configuration
- Collection aliases: Custom aliases for collections in GraphQL schema
- Schema automatically updates when configuration changes

### Typesense Integration Submodule

The `typesense_integration` submodule provides helpful Search API processors:

- **Breadcrumbs processor** (`typesense_breadcrumb_text`): Indexes breadcrumb text for nodes
- **Entity URL processor** (`typesense_entity_url`): Indexes canonical URLs for entities (supports Media entities)
- **HTML Filter Safe processor** (`html_filter_safe`): Improved HTML filtering that preserves word boundaries across block elements
- **Remove Excluded processor** (`typesense_remove_excluded`): Filters out entities flagged to be excluded from search
- **Schema Alter Subscriber**: Automatically sets locale for all fields based on site's default language

### Performance Optimizations

- **Conditional facet loading**: Facets are only returned when explicitly requested via `facetFields`
- **Field selection**: Include/exclude specific fields to reduce payload size
- **Zero-hit queries**: When hits are not requested, sets `per_page` to 0 for facet-only queries
- **Efficient filtering**: Smart handling of null/empty facet filters

### KeyValue Facets

KeyValue facets allow you to create facets from objects with `id` and `label` properties. This is useful for cases like page bundles where you need both a machine name (key) and a human-readable label.

#### Setting up a KeyValue Facet

1. **Create a Search API Processor** that provides KeyValue data:

```php
<?php

namespace Drupal\your_module\Plugin\search_api\processor;

use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\search_api\Processor\ProcessorProperty;
use Drupal\search_api\Item\ItemInterface;

/**
 * @SearchApiProcessor(
 *   id = "your_keyvalue_processor",
 *   label = @Translation("Your KeyValue Processor"),
 *   stages = {
 *     "add_properties" = 0,
 *   },
 * )
 */
class YourKeyValueProcessor extends ProcessorPluginBase {

  public function getPropertyDefinitions($datasource = NULL) {
    $properties = [];
    if (!$datasource) {
      $properties['your_field'] = new ProcessorProperty([
        'label' => $this->t('Your Field'),
        'type' => 'string',
        'is_list' => FALSE,
        'processor_id' => $this->getPluginId(),
      ]);
    }
    return $properties;
  }

  public function addFieldValues(ItemInterface $item) {
    $entity = $item->getOriginalObject()->getValue();
    
    // Your logic to determine id and label
    $id = 'machine_name';
    $label = 'Human Readable Label';
    
    $fields = $this->getFieldsHelper()
      ->filterForPropertyPath($item->getFields(), NULL, 'your_field');
    
    foreach ($fields as $field) {
      $field->addValue([
        'id' => $id,
        'label' => $label,
      ]);
    }
  }
}
```

2. **Configure the Search API Field**:
   - Add the processor to your Search API index
   - Add a field using the processor property
   - Set the field data type to **"Typesense: Key Value"** (`typesense_key_value[]`)
   - Enable faceting on the field

3. **Use in GraphQL Queries**:

```graphql
query {
  searchTypesense(
    collection: CONTENT
    text: "search term"
    facetFields: [CONTENT_YOUR_FIELD]
    fulltextFields: [{ name: "title", weight: 2 }]
  ) {
    facets {
      id
      type
      terms {
        id
        label
        count
      }
    }
    hits {
      document {
        your_field {
          id
          label
        }
      }
    }
  }
}
```

The KeyValue facet will automatically:
- Use the `.id` subfield for faceting
- Return full objects with `id` and `label` via `facet_return_parent`
- Expose the facet enum without the `___ID` suffix (e.g., `CONTENT_YOUR_FIELD` instead of `CONTENT_YOUR_FIELD___ID`)
- Make the field queryable as an array of `TypesenseKeyValue` objects in document types

### Example Queries

See `example-queries.graphql` for comprehensive examples of:

- Basic searches
- Faceted searches with filtering
- Hybrid search with semantic features
- Highlighting and text match info
- Field selection and pagination

## MAINTAINERS

Current maintainers for Drupal 11:

- ayalon - https://www.drupal.org/u/ayalon
