<?php

declare(strict_types=1);

namespace Drupal\typesense_integration\EventSubscriber;

use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\search_api_typesense\Event\TypesenseSchemaEvent;
use Drupal\search_api_typesense\Event\TypesenseSchemaEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to Typesense schema alter events to adjust field locales.
 */
final class TypesenseSchemaAlterSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a new subscriber.
   */
  public function __construct(
    private readonly LanguageManagerInterface $languageManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      TypesenseSchemaEvents::ALTER_SCHEMA => 'onAlterSchema',
    ];
  }

  /**
   * Alters the generated Typesense schema to add locale to all fields.
   */
  public function onAlterSchema(TypesenseSchemaEvent $event): void {
    $generated = $event->getGeneratedSchema();

    if (!is_array($generated) || empty($generated['fields']) || !is_array($generated['fields'])) {
      return;
    }

    $default_langcode = $this->languageManager->getDefaultLanguage()->getId();

    foreach ($generated['fields'] as &$field) {
      if (is_array($field)) {
        // Set or override the field's locale to the site's default language.
        $field['locale'] = $default_langcode;
      }
    }
    unset($field);

    $event->setGeneratedSchema($generated);
  }

}
