<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql;

use Drupal\typesense_graphql\Enum\GraphQlEnumTypeInterface;
use Drupal\graphql_core_schema\SchemaBuilder\SchemaBuilderGenerator;
use GraphQL\Type\Definition\EnumType;

/**
 * Provides reusable code for defining GraphQL enums.
 *
 * Can be used in schema extension plugins.
 */
trait EnumDefinitionsTrait {

  /**
   * Adds the given enums to the GraphQL schema builder generator.
   *
   * @param \Drupal\graphql_core_schema\SchemaBuilder\SchemaBuilderGenerator $generator
   *   The GraphQL schema generator to add the enums to.
   * @param array $enumsToExpose
   *   An associative array of enums to expose, where the keys are GraphQL enum
   *   names and the values are an associative array with the following keys:
   *   - 'enum': The enum class.
   *   - 'description': The description of the enum.
   *   - 'hasLabels': (optional) A boolean indicating if the enum has labels.
   *   - 'backed': (optional) A boolean indicating if the enum is backed.
   */
  protected function addEnumDefinitions(SchemaBuilderGenerator $generator, array $enumsToExpose): void {
    foreach ($enumsToExpose as $type => $enum) {
      $values = [];
      foreach ($enum['enum']::cases() as $case) {
        $case_name = match (TRUE) {
          $case instanceof GraphQlEnumTypeInterface => $case->toGraphQlEnumValue(),
          $enum['backed'] => $case->value,
          default => $case->name,
        };
        $value = ['value' => $enum['backed'] ? $case->value : $case_name];
        if ($enum['hasLabels']) {
          $value['description'] = (string) $case->label();
        }
        $values[$case_name] = $value;
      }

      $generator->addType(new EnumType([
        'name' => $type,
        'description' => $enum['description'],
        'values' => $values,
      ]));
    }
  }

}
