<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Model;

/**
 * Helper object for a Typesense "multiSearch" response.
 *
 * In a multiSearch query, we esentially perform multiple search queries in a
 * single request. Each search query is completely independent.
 *
 * In our setup, we use the multiSearch feature to get the results in the first
 * search query and then add a query for every possible facet. This is done so
 * that we can get correct facet counts.
 */
final class TypesenseMultiSearchResponse {

  /**
   * An array of multiSearch search results.
   */
  protected array $multiSearchResults;

  /**
   * Constructs a new TypesenseMultiSearchResponse.
   *
   * @param array $response
   *   The multiSearch response.
   * @param \Drupal\typesense_graphql\Model\TypesenseQuery $query
   *   The query used for the search.
   */
  public function __construct(
    array $response,
    protected ?TypesenseQuery $query = NULL,
  ) {
    // Contains an array of search responses.
    $results = $response['results'] ?? NULL;

    if (empty($results)) {
      throw new \RuntimeException('Unexpected empty response from Typesense.');
    }

    // Check for any errors in the responses.
    $errors = [];
    foreach ($results as $result) {
      if (isset($result['error'])) {
        $errors[] = $result['error'];
      }
    }

    // Throw an exception if we have any errors.
    // When using multiSearch, it's possible to get a 200 response, however
    // each individual search could itself contain an error. Since we need
    // all queries to be successful, we need to throw an exception if just
    // one query has an error.
    if (!empty($errors)) {
      throw new \RuntimeException(
        'Typesense multiSearch errors: ' . implode('; ', $errors)
      );
    }

    $this->multiSearchResults = $results;
  }

  /**
   * Creates a new empty multiSearch response.
   *
   * Can be used to simulate a valid response with 0 results.
   *
   * @return static
   */
  public static function createEmpty(): static {
    return new static([
      'results' => [
        [
          'hits' => [],
          'found' => 0,
        ],
      ],
    ]);
  }

  /**
   * Get all search results.
   *
   * @return array
   *   The search results
   */
  public function getSearchResults(): array {
    return $this->multiSearchResults;
  }

  /**
   * Get the hits.
   *
   * @return array
   *   The hits.
   */
  public function getHits(): array {
    // In our multiSearch query, the first search is always the results.
    // Any other queries are used for building facet counts.
    $firstResult = $this->multiSearchResults[0] ?? [];

    // If group_by is enabled, hits are under 'grouped_hits' instead of 'hits'.
    if (isset($firstResult['grouped_hits'])) {
      $hits = [];
      foreach ($firstResult['grouped_hits'] as $group) {
        // Each group contains its own 'hits' array.
        if (isset($group['hits']) && is_array($group['hits'])) {
          $hits = array_merge($hits, $group['hits']);
        }
      }
      return $hits;
    }

    return $firstResult['hits'] ?? [];
  }

  /**
   * Get the total.
   *
   * @return int
   *   The total number of hits.
   */
  public function getTotal(): int {
    return $this->multiSearchResults[0]['found'] ?? 0;
  }

  /**
   * Get the query used for the request.
   *
   * @return \Drupal\typesense_graphql\Model\TypesenseQuery|null
   *   The query.
   */
  public function getQuery(): ?TypesenseQuery {
    return $this->query;
  }

}
