<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Model;

use Drupal\search_api\IndexInterface;
use Drupal\typesense_graphql\Enum\TypesenseSearchMode;

/**
 * Helper object for constructing a Typesense multiSearch query.
 */
class TypesenseQuery {

  /**
   * Which document fields to include in the response.
   *
   * @var string[]
   */
  protected array $includeFields = [];

  /**
   * Which document fields to highlight.
   *
   * @var string[]
   */
  protected array $highlightFields = [];

  /**
   * Which document fields to highlight fully without snippeting.
   *
   * @var string[]
   */
  protected array $highlightFullFields = [];

  /**
   * Filters.
   *
   * @var string[]
   */
  protected array $filters = [];

  /**
   * The optional search text; if null, defaults to '*' (match all).
   *
   * @var string|null
   */
  protected ?string $searchText = NULL;

  /**
   * How to sort the results.
   *
   * @var string|null
   */
  protected ?string $sortBy = NULL;

  /**
   * The vector query.
   *
   * @var string|null
   */
  protected ?string $vectorQuery = NULL;

  /**
   * The search mode.
   *
   * @var string
   */
  protected string $searchMode;

  /**
   * The alpha parameter for hybrid search.
   *
   * @var float
   */
  protected float $alpha = 0.8;

  /**
   * The distance threshold for vector search.
   *
   * @var float
   */
  protected float $distanceThreshold = 0.5;

  /**
   * Currently selected facets to filter on.
   *
   * Array of ['id' => facetField, 'values' => [value1, ...]].
   *
   * @var array
   */
  protected array $selectedFacets = [];

  /**
   * The fields to use for full text search.
   *
   * @var \Drupal\typesense_graphql\Model\TypesenseQueryFulltextField[]
   */
  protected array $fulltextFields = [];


  /**
   * Search param overrides.
   */
  protected array $searchParamOverrides = [];

  /**
   * Number of results per page.
   *
   * @var int
   */
  protected int $perPage = 32;

  /**
   * Current page index (0-based for our logic).
   *
   * @var int
   */
  protected int $page = 0;

  /**
   * Indicator whether the query can be skipped.
   */
  protected bool $shouldSkip = FALSE;

  /**
   * The alias for the index.
   */
  protected string $alias = '';

  /**
   * Facet fields that are entity reference facets (require facet_return_parent).
   *
   * These include taxonomy term facets and node reference facets.
   *
   * @var string[]
   */
  protected array $entityReferenceFacetFields = [];

  /**
   * Constructs a new TypesenseQuery object.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Typesense collection.
   * @param string[] $facetFields
   *   The list of facet field names to request.
   * @param string[] $entityReferenceFacetFields
   *   The list of facet field names that are entity reference facets
   *   (taxonomy term or node reference).
   */
  public function __construct(
    private IndexInterface $index,
    private array $facetFields = [],
    array $entityReferenceFacetFields = [],
  ) {
    $this->entityReferenceFacetFields = $entityReferenceFacetFields;
    $this->searchMode = TypesenseSearchMode::KeywordSearch->toGraphQlEnumValue();
  }

  /**
   * Mark the query as skippable.
   *
   * @return static
   */
  public function markSkippable(): static {
    $this->shouldSkip = TRUE;
    return $this;
  }

  /**
   * Whether the query is skippable.
   *
   * @return bool
   *   TRUE if the query is skippable.
   */
  public function isSkippable(): bool {
    return $this->shouldSkip;
  }

  /**
   * Get the collection.
   *
   * @return string
   *   The collection.
   */
  public function getCollection(): string {
    return $this->index->getServerInstance()->getBackend()->getCollectionName($this->index);
  }

  /**
   * Set the alias for the collection.
   *
   * @param string $alias
   *   The alias to set.
   *
   * @return static
   */
  public function setAlias(string $alias): static {
    $this->alias = $alias;
    return $this;
  }

  /**
   * Get the alias for the collection.
   *
   * If no alias is set, returns the collection name.
   *
   * @return string
   *   The alias or collection name.
   */
  public function getAlias(): string {
    return $this->alias ?: $this->getCollection();
  }

  /**
   * Get the index.
   *
   * @return \Drupal\search_api\IndexInterface
   *   The index.
   */
  public function getIndex(): IndexInterface {
    return $this->index;
  }

  /**
   * Sets the available facet fields.
   *
   * @param string[] $facetFields
   *   The facet fields.
   *
   * @return static
   */
  public function setFacetFields(array $facetFields): static {
    $this->facetFields = $facetFields;
    return $this;
  }

  /**
   * Sets the fields included in the hits.
   *
   * @param string[]|string $includeFields
   *   The fields or field to include.
   *
   * @return static
   */
  public function addIncludeFields(array|string $includeFields): static {
    if (is_string($includeFields)) {
      $this->includeFields[] = $includeFields;
    }
    else {
      $this->includeFields = array_merge($this->includeFields, $includeFields);
    }
    return $this;
  }

  public function addHighlightFields(array|string $highlightFields): static {
    if (is_string($highlightFields)) {
      $this->highlightFields[] = $highlightFields;
    }
    else {
      $this->highlightFields = array_merge($this->highlightFields, $highlightFields);
    }
    return $this;
  }

  /**
   * Add fields to highlight fully without snippeting.
   *
   * @param array|string $highlightFullFields
   *   The fields to highlight fully.
   *
   * @return static
   *   The query object for chaining.
   */
  public function addHighlightFullFields(array|string $highlightFullFields): static {
    if (is_string($highlightFullFields)) {
      $this->highlightFullFields[] = $highlightFullFields;
    }
    else {
      $this->highlightFullFields = array_merge($this->highlightFullFields, $highlightFullFields);
    }
    return $this;
  }

  /**
   * Add a fulltext field.
   *
   * @param string $name
   *   The name of the field.
   *
   * @return \Drupal\typesense_graphql\Model\TypesenseQueryFulltextField
   *   The fulltext field object.
   */
  public function defineFulltextField(string $name): TypesenseQueryFulltextField {
    $field = new TypesenseQueryFulltextField($name);
    $this->fulltextFields[] = $field;
    return $field;
  }

  /**
   * Sets the search text.
   *
   * @param string $text
   *   The text to search for.
   *
   * @return static
   */
  public function setSearchText(string $text): static {
    $this->searchText = $text;
    return $this;
  }

  /**
   * Set the search mode.
   *
   * @param string $searchMode
   *   The search mode: 'KEYWORD_SEARCH', 'SEMANTIC_SEARCH', or 'HYBRID_SEARCH'.
   *
   * @return static
   *   The current object (for chaining).
   */
  public function setSearchMode(string $searchMode): static {
    $this->searchMode = $searchMode;
    return $this;
  }

  /**
   * Get the search mode.
   *
   * @return string
   *   The search mode.
   */
  public function getSearchMode(): string {
    return $this->searchMode;
  }

  /**
   * Set the alpha parameter for hybrid search.
   *
   * @param float $alpha
   *   The alpha value (0.0 to 1.0).
   *
   * @return static
   *   The current object (for chaining).
   */
  public function setAlpha(float $alpha): static {
    $this->alpha = $alpha;
    return $this;
  }

  /**
   * Get the alpha parameter.
   *
   * @return float
   *   The alpha value.
   */
  public function getAlpha(): float {
    return $this->alpha;
  }

  /**
   * Set the distance threshold for vector search.
   *
   * @param float $distanceThreshold
   *   The distance threshold.
   *
   * @return static
   *   The current object (for chaining).
   */
  public function setDistanceThreshold(float $distanceThreshold): static {
    $this->distanceThreshold = $distanceThreshold;
    return $this;
  }

  /**
   * Get the distance threshold.
   *
   * @return float
   *   The distance threshold.
   */
  public function getDistanceThreshold(): float {
    return $this->distanceThreshold;
  }

  /**
   * Set the vector query.
   *
   * Fair dice roll has chosen the alpha and distanceThreshold default values.
   *
   * @param string $field
   *   The field name.
   * @param string $vectors
   *   The comma separated vectors. If empty, TypeSense will automatically generate the empbedding from the q param.
   * @param float $alpha
   *   The ratio how keyword and semantic scores should be weighted.
   * @param float $distanceThreshold
   *   The similarity threshold.
   *
   * @return static
   *   The current object (for chaining).
   */
  public function setVectorQuery(string $field, string $vectors = '', float $alpha = 0.8, float $distanceThreshold = 0.5): static {
    $this->vectorQuery = "$field:([$vectors], alpha: $alpha, distance_threshold: $distanceThreshold, k: 100)";
    return $this;
  }

  /**
   * Set the similarity vector query.
   *
   * @param string $field
   *   The field name.
   * @param string $id
   *   The document ID.
   *
   * @return static
   *   The current object (for chaining).
   */
  public function setSimilarityVectorQuery(string $field, string $id): static {
    $this->vectorQuery = sprintf("%s:([], id: %s)", $field, $id);
    return $this;
  }

  /**
   * Sets the selected facets to filter results.
   *
   * @param array $facets
   *   Array of facets, each element an array with 'id' and 'values'.
   *
   * @return static
   */
  public function setSelectedFacets(array $facets): static {
    $this->selectedFacets = $facets;
    return $this;
  }

  /**
   * Sets the number of results to return per page.
   *
   * @param int $perPage
   *   Number of results per page.
   *
   * @return static
   */
  public function setPerPage(int $perPage): static {
    $this->perPage = $perPage;
    return $this;
  }

  /**
   * Sets the page index for paginated results.
   *
   * @param int $page
   *   The page index (0-based).
   *
   * @return static
   */
  public function setPage(int $page): static {
    $this->page = $page;
    return $this;
  }

  /**
   * Add a filter.
   *
   * @param string $filter
   *   The filter.
   *
   * @return static
   */
  public function addFilter(string $filter): static {
    $this->filters[] = $filter;
    return $this;
  }

  /**
   * Add a boolean filter.
   *
   * @param string $fieldName
   *   The field name.
   * @param bool $value
   *   The value.
   *
   * @return static
   */
  public function addBooleanFilter(string $fieldName, bool $value): static {
    $boolValue = $value ? 'true' : 'false';
    $this->addFilter("$fieldName:=$boolValue");
    return $this;
  }

  /**
   * Add an equals filter.
   *
   * @param string $fieldName
   *   The field name.
   * @param string|int $value
   *   The value.
   *
   * @return static
   */
  public function addEqualsFilter(string $fieldName, string|int $value): static {
    $this->addFilter("$fieldName:=$value");
    return $this;
  }

  /**
   * Add a contains filter.
   *
   * @param string $fieldName
   *   The field name.
   * @param string[]|int[] $values
   *   The values.
   *
   * @return static
   */
  public function addContainsFilter(string $fieldName, array $values): static {
    $filterValue = implode(',', $values);
    $this->addFilter("$fieldName:[$filterValue]");
    return $this;
  }

  /**
   * Add a numeric range filter.
   *
   * @param string $fieldMin
   *   The field name for the minimum value.
   * @param string $fieldMax
   *   The field name for the maximum value.
   * @param int|string $value
   *   The value.
   *
   * @return static
   */
  public function addRangeFilter(string $fieldMin, string $fieldMax, int|string $value): static {
    $this->addFilter("($fieldMax:>=$value && $fieldMin:<=$value)");
    return $this;
  }

  /**
   * Builds the full payload array for Typesense multiSearch.
   *
   * @return array
   *   An associative array suitable for $client->multiSearch->perform().
   */
  public function toQuery(): array {
    return ['searches' => $this->buildSearchRequests()];
  }

  /**
   * Builds the array of individual search queries.
   *
   * The first query returns hits and all facets;
   * subsequent queries (one per active facet) return only facet counts.
   *
   * @return array
   *   List of search query definitions.
   */
  protected function buildSearchRequests(): array {
    $requests = [];

    // Main query: hits + all facets.
    $requests[] = $this->buildSingleRequest();

    // Additional queries: No hits for each selected facet.
    foreach ($this->activeFacetIds() as $id) {
      $requests[] = $this->buildSingleRequest($id);
    }

    return $requests;
  }

  /**
   * Builds the facet query configuration.
   *
   * @param array $facetIds
   *   The facet IDs to build the query for.
   *
   * @return array
   *   The facet query configuration.
   */
  protected function buildFacetQuery(array $facetIds): array {
    $facetBy = [];
    $facetReturnParent = [];

    foreach ($facetIds as $id) {
      $facetBy[] = $id;
      // If this is an entity reference facet (taxonomy term or node reference),
      // add it to facet_return_parent.
      if (in_array($id, $this->entityReferenceFacetFields, TRUE)) {
        $facetReturnParent[] = $id;
      }
    }
    $query = [
      'facet_by' => implode(',', $facetBy),
      'facet_strategy' => 'exhaustive',
      'max_facet_values' => 1000,
    ];

    if (!empty($facetReturnParent)) {
      $query['facet_return_parent'] = implode(',', $facetReturnParent);
    }

    return $query;
  }

  /**
   * Builds a single search query definition.
   *
   * @param string|null $skipFacetId
   *   If provided, exclude this facet from filters (to compute its counts).
   *
   * @return array
   *   The query definition.
   */
  protected function buildSingleRequest(?string $skipFacetId = NULL): array {
    $mergeFulltextProperties = function (string $property) {
      $values = [];

      foreach ($this->fulltextFields as $field) {
        $value = $field->{$property};
        if (is_bool($value)) {
          $values[] = $value ? 'true' : 'false';
        }
        else {
          $values[] = $value;
        }
      }

      return implode(',', $values);
    };

    $base = array_merge([
      'collection' => $this->getCollection(),
      'prioritize_exact_match' => TRUE,
      'prioritize_token_position' => TRUE,
      'rerank_hybrid_matches' => TRUE,
      'filter_by' => $this->buildFilterBy($skipFacetId),
      'synonym_prefix' => FALSE,
      'drop_tokens_threshold' => 0,
      'typo_tokens_threshold' => 1,
      // 'text_match_type' => 'sum_score', @todo: What is this for?
      'include_fields' => implode(',', $this->includeFields),
      'highlight_fields' => implode(',', $this->highlightFields),
      'highlight_affix_num_tokens' => 20,
    ], $this->searchParamOverrides);

    // Add highlight_full_fields if there are any full highlight fields.
    if (!empty($this->highlightFullFields)) {
      $base['highlight_full_fields'] = implode(',', $this->highlightFullFields);
    }

    // Exclude embedding field if embeddings are enabled.
    $backend = $this->index->getServerInstance()->getBackend();
    $schema = $backend->getSchemaForIndex($this->index);
    $embeddingsEnabled = $schema && $schema->isEmbeddingEnabled();
    if ($embeddingsEnabled) {
      $base['exclude_fields'] = 'embedding';
      $base['group_by'] = 'document_id';
      $base['group_limit'] = '1';
    }

    if ($this->searchText) {
      $base['q'] = $this->searchText;

      // Handle different search modes.
      if ($this->searchMode === TypesenseSearchMode::SemanticSearch->toGraphQlEnumValue()) {
        // Semantic search: only query by embedding.
        if (!$embeddingsEnabled) {
          throw new \RuntimeException('Semantic search mode requires embeddings to be enabled on the collection.');
        }
        // Set vector query - Typesense will automatically generate embedding from q param.
        $this->setVectorQuery('embedding', '', $this->alpha, $this->distanceThreshold);
        $base['vector_query'] = $this->vectorQuery;
        $base['query_by'] = 'embedding';
        unset($base['query_by_weights']);
        unset($base['num_typos']);
        $base['prefix'] = FALSE;
        unset($base['infix']);
      }
      elseif ($this->searchMode === TypesenseSearchMode::HybridSearch->toGraphQlEnumValue()) {
        // Hybrid search: query by both embedding and fulltext fields.
        if (!$embeddingsEnabled) {
          throw new \RuntimeException('Hybrid search mode requires embeddings to be enabled on the collection.');
        }
        // Set vector query - Typesense will automatically generate embedding from q param.
        $this->setVectorQuery('embedding', '', $this->alpha, $this->distanceThreshold);
        $base['vector_query'] = $this->vectorQuery;

        // Combine embedding with fulltext fields.
        $fulltextFields = $mergeFulltextProperties('name');
        if (!empty($fulltextFields)) {
          $base['query_by'] = 'embedding,' . $fulltextFields;
          // Prepend embedding values: weight=1, num_typos=0, prefix=false, infix=off
          // These must match the number of fields in query_by (embedding + fulltext fields)
          $base['query_by_weights'] = '1,' . $mergeFulltextProperties('weight');
          $base['num_typos'] = '0,' . $mergeFulltextProperties('numTypos');
          $base['synonym_num_typos'] = 0;
          // Prefix and infix: embedding doesn't support these, so use false/off
          // Always prepend embedding value, then append fulltext field values
          $fulltextPrefix = $mergeFulltextProperties('prefix');
          $base['prefix'] = 'false,' . $fulltextPrefix;
          $fulltextInfix = $mergeFulltextProperties('infix');
          $base['infix'] = 'off,' . $fulltextInfix;
        }
        else {
          // Fallback to semantic search if no fulltext fields are defined.
          $base['query_by'] = 'embedding';
          unset($base['query_by_weights']);
          unset($base['num_typos']);
          $base['prefix'] = FALSE;
          unset($base['infix']);
        }
      }
      else {
        // Keyword search (default): query by fulltext fields only.
        $base['query_by'] = $mergeFulltextProperties('name');
        $base['query_by_weights'] = $mergeFulltextProperties('weight');
        $base['num_typos'] = $mergeFulltextProperties('numTypos');
        $base['synonym_num_typos'] = 0;
        $base['prefix'] = $mergeFulltextProperties('prefix');
        $base['infix'] = $mergeFulltextProperties('infix');
      }
    }
    else {
      $base['q'] = '*';
    }

    // Handle legacy vector query if set (for backward compatibility).
    if ($this->vectorQuery && $this->searchMode === TypesenseSearchMode::KeywordSearch->toGraphQlEnumValue()) {
      $base['vector_query'] = $this->vectorQuery;
      // If vector query is set in keyword mode, override query_by to embedding only.
      $base['query_by'] = 'embedding';
      unset($base['query_by_weights']);
      unset($base['num_typos']);
      $base['prefix'] = FALSE;
      unset($base['infix']);
    }

    if ($skipFacetId) {
      // Only generate facets for the facet ID being skipped.
      $base = array_merge($base, $this->buildFacetQuery([$skipFacetId]));
      $base['per_page'] = 0;
      $base['page'] = 1;
    }
    else {
      // Only generate facets if facet fields are requested.
      if (!empty($this->facetFields)) {
        $base = array_merge($base, $this->buildFacetQuery($this->facetFields));
      }
      $base['per_page'] = $this->perPage;
      $base['page'] = $this->page;
      $sortBy = $this->buildSortBy();
      if ($sortBy) {
        $base['sort_by'] = $sortBy;
      }
    }

    return $base;
  }

  /**
   * Builds the filter_by string.
   *
   * @param string|null $skipId
   *   If provided, do not include this facet in the filter list.
   *
   * @return string
   *   The filter_by clause.
   */
  protected function buildFilterBy(?string $skipId): string {
    $filters = [...$this->filters];

    foreach ($this->selectedFacets as $facet) {
      $facetId = $facet['id'];
      if ($facetId === $skipId) {
        continue;
      }
      // Skip facets with empty values arrays or null values - treat as if no facet filter was provided.
      if (empty($facet['values']) || empty(array_filter($facet['values'], fn($v) => $v !== NULL))) {
        continue;
      }
      $values = implode(',', $facet['values']);
      $filters[] = sprintf('%s:[%s]', $facet['id'], $values);
    }

    return implode(' && ', $filters);
  }

  /**
   * Returns the IDs of currently applied facet filters.
   *
   * @return string[]
   *   Unique list of facet IDs.
   */
  protected function activeFacetIds(): array {
    return array_unique(array_map(fn($f) => $f['id'], $this->selectedFacets));
  }

  /**
   * Sets the sort by clause.
   *
   * @param string $sortBy
   *   The sort by clause.
   *
   * @return static
   *   The query object for method chaining.
   */
  public function setSortBy(string $sortBy): static {
    $this->sortBy = $sortBy;
    return $this;
  }

  /**
   * Builds the sort_by clause.
   *
   * If coordinates are set, sorts by ascending distance; otherwise by
   * date_from.
   *
   * @return string|null
   *   The sort_by definition.
   */
  protected function buildSortBy(): string|null {
    return $this->sortBy;
  }

  /**
   * Set a search param.
   *
   * @param string $key
   *   The key.
   * @param mixed $value
   *   The value.
   *
   * @return static
   */
  public function setSearchParam(string $key, $value): static {
    $this->searchParamOverrides[$key] = $value;
    return $this;
  }

}
