<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Resolves the term map for the search.
 *
 * @DataProducer(
 *   id = "facet_term_labels",
 *   name = @Translation("Facet Term Labels"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("A map of ID to entity labels")
 *   ),
 *   consumes = {
 *     "langcode" = @ContextDefinition("string",
 *       label = @Translation("The current langcode."),
 *     ),
 *   }
 * )
 */
class FacetTermLabels extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $pluginId,
    $pluginDefinition,
  ) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Constructs a new FacetTermLabels producer.
   *
   * @param array<mixed> $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin id.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManager $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * Resolves the facet term labels.
   *
   * @param string $langcode
   *   The current langcode.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $fieldContext
   *   The GraphQL field context.
   *
   * @return array<string, string>
   *   The search terms map.
   */
  public function resolve(string $langcode, FieldContext $fieldContext): array {
    // A map of entity types and their bundles.
    /** @var array<string, string[]> $entityTypeBundles */
    $entityTypeBundles = [];

    // A map of bundles to their corresponding facet ID.
    /** @var array<string, string> $bundleToFacetId */
    $bundleToFacetId = [];

    // @todo adapt
    // phpcs:ignore @phpstan-ignore-next-line
    foreach (TypesenseFacetId::cases() as $facetId) {
      $providing = $facetId->getProvidingEntityTypeAndBundle();
      if (!empty($providing)) {
        $entityTypeId = $providing[0];
        $bundle = $providing[1];

        $entityTypeBundles[$entityTypeId][] = $bundle;
        $bundleToFacetId[$bundle] = $facetId->toGraphQlEnumValue();

        // Add the bundle-specific list cache tag.
        $fieldContext->addCacheTags([$entityTypeId . '_list:' . $bundle]);
      }
    }

    // The final map. The keys are FACET_ID:ENTITY_ID and the values are the
    // entity labels.
    /** @var array<string, string> $facetTermLabelMap */
    $facetTermLabelMap = [];

    foreach ($entityTypeBundles as $entityTypeId => $bundles) {
      $storage = $this->entityTypeManager->getStorage($entityTypeId);
      $query = $storage->getQuery()->accessCheck(TRUE);

      if ($entityBundleKey = $storage->getEntityType()->getKey('bundle')) {
        $query->condition($entityBundleKey, $bundles, 'IN');
      }

      if ($publishedKey = $storage->getEntityType()->getKey('published')) {
        $query->condition($publishedKey, '1');
      }

      $ids = $query->execute();

      foreach (array_chunk($ids, 50) as $batch) {
        $entities = $storage->loadMultiple($batch);
        foreach ($entities as $entity) {
          $label = NULL;

          if (
            $entity instanceof TranslatableInterface &&
            $entity->isTranslatable() &&
            $entity->hasTranslation($langcode)
          ) {
            $label = (string) $entity->getTranslation($langcode)->label();
          }
          else {
            $label = (string) $entity->label();
          }

          if (!$label) {
            continue;
          }

          $bundle = $entity->bundle();
          $entityId = $entity->id();
          $facetId = $bundleToFacetId[$bundle];
          $key = "$facetId:$entityId";
          $facetTermLabelMap[$key] = (string) $entity->label();
        }
      }
    }

    return $facetTermLabelMap;
  }

}
