<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL\DataProducer;

use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\typesense_graphql\Enum\TypesenseFacetType;
use Drupal\typesense_graphql\Model\TypesenseMultiSearchResponse;
use Drupal\typesense_graphql\TypesenseCollectionManager;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Produces the facets.
 *
 * Supports three types of facets:
 * 1. Normal string facets: Simple key-value pairs with counts.
 * 2. Hierarchical taxonomy facets: Facets with parent-child relationships.
 * 3. KeyValue facets: Facets with key and value properties (e.g., Page bundles).
 *
 * For hierarchical facets, we assume they are indexed using the
 * "entity_reference_object" processor, which indexes an object instead of
 * just the ID:
 *
 * [
 *   'id' => '123',
 *   'label' => 'Foobar',
 *   'parent' => '789',
 *   'weight' => -40,
 * ]
 *
 * If "facet_return_parent" is provided in the Typesense query, each
 * facet count will contain a "parent" property with the full object.
 * That way we can reconstruct the entire hierarchy.
 *
 * For KeyValue facets, the values are objects with "id" and "label" properties:
 * [
 *   'id' => 'machine_name',
 *   'label' => 'Label',
 * ]
 *
 * @DataProducer(
 *   id = "typesense_graphql_facets",
 *   name = "Search Typesense Facets",
 *   description = @Translation("Produce the facets"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The facets")
 *   ),
 *   consumes = {
 *     "response" = @ContextDefinition("any",
 *       label = @Translation("The Typesense multiSearch response"),
 *     ),
 *   }
 * )
 */
final class TypesenseFacets extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The Typesense collection manager.
   *
   * @var \Drupal\typesense_graphql\TypesenseCollectionManager
   */
  protected TypesenseCollectionManager $collectionManager;

  /**
   * Constructs a new TypesenseFacets data producer.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\typesense_graphql\TypesenseCollectionManager $collection_manager
   *   The Typesense collection manager.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    TypesenseCollectionManager $collection_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->collectionManager = $collection_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('typesense_graphql.collection_manager')
    );
  }

  /**
   * Resolves the search results.
   *
   * @param \Drupal\typesense_graphql\Model\TypesenseMultiSearchResponse $response
   *   A Typesense multiSearch response.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $fieldContext
   *   The GraphQL field context, which contains the field definition.
   *
   * @return array
   *   The mapped, nested facets.
   */
  protected function resolve(TypesenseMultiSearchResponse $response, FieldContext $fieldContext): array {
    $results = $response->getSearchResults();

    // This can be the machine name of the index or an alias.
    $collection = $fieldContext->getContextValue('typesense_graphql_collection');

    // Get the index to check field types.
    $collectionNormalized = strtoupper(str_replace('-', '_', $collection));
    $index = $this->collectionManager->getIndexByCollectionName($collectionNormalized);
    $indexFields = $index ? $index->getFields(TRUE) : [];

    // Use getAliasForIndexId() to ensure consistency with enum building and handle empty strings correctly.
    $collectionAlias = $index ? $this->collectionManager->getAliasForIndexId($index->id()) : $collection;

    $facets = [];

    foreach ($results as $result) {
      foreach ($result['facet_counts'] as $facetRaw) {
        $facetId = $facetRaw['field_name'];

        // Make sure we only produce known facet keys.
        if (!$facetId) {
          // @todo Would it be better to throw an error here?
          continue;
        }

        // Check if this is a KeyValue facet by checking the field type.
        // Check both the facet field itself and if it's a subfield, check the parent field.
        $isKeyValue = FALSE;
        if (isset($indexFields[$facetId])) {
          $fieldType = $indexFields[$facetId]->getType();
          if ($fieldType === 'typesense_key_value[]') {
            $isKeyValue = TRUE;
          }
        }
        // Check if this is a subfield of a KeyValue field (e.g., "field_bundle.id").
        if (!$isKeyValue && strpos($facetId, '.id') !== FALSE) {
          $parentFieldName = substr($facetId, 0, strpos($facetId, '.id'));
          if (isset($indexFields[$parentFieldName])) {
            $parentFieldType = $indexFields[$parentFieldName]->getType();
            if ($parentFieldType === 'typesense_key_value[]') {
              $isKeyValue = TRUE;
            }
          }
        }

        // Check if this is a hierarchical facet by looking for parent info.
        // For all entity reference facets, we use the TypeSense option "facet_return_parent" to return the parents.
        $isHierarchical = FALSE;
        if (!$isKeyValue) {
          foreach ($facetRaw['counts'] as $count) {
            if (isset($count['parent']) && is_array($count['parent'])) {
              $isHierarchical = TRUE;
              break;
            }
          }
        }

        if ($isKeyValue) {
          // Handle KeyValue facets.
          $terms = $this->processKeyValueFacet($facetRaw['counts']);
        }
        elseif ($isHierarchical) {
          // Handle hierarchical facets (taxonomy terms).
          $terms = $this->processHierarchicalFacet($facetRaw['counts']);
        }
        else {
          // Handle normal string facets.
          $terms = $this->processStringFacet($facetRaw['counts']);
        }

        // Format the facet ID to match the enum format:
        // 1. Convert collection to uppercase and replace hyphens with underscores
        // 2. Convert facetId to uppercase, replace hyphens with underscores, and dots with triple underscores
        // For KeyValue facets, strip the .id suffix to match the enum key.
        $enumFacetId = $facetId;
        if ($isKeyValue && strpos($facetId, '.id') !== FALSE) {
          $enumFacetId = substr($facetId, 0, strpos($facetId, '.id'));
        }
        // Use the collection alias (or index_id if no alias) for consistency with enum building.
        $collectionKey = strtoupper(str_replace('-', '_', $collectionAlias));
        $facetKey = strtoupper(str_replace(['-', '.'], ['_', '___'], $enumFacetId));

        // Ensure valid enum name (cannot start with a number)
        if (is_numeric($facetKey[0])) {
          $facetKey = 'FACET_' . $facetKey;
        }

        $enumKey = $collectionKey . '_' . $facetKey;

        // Determine the facet type.
        if ($isKeyValue) {
          $facetType = TypesenseFacetType::KeyValue->toGraphQlEnumValue();
        }
        elseif ($isHierarchical) {
          $facetType = TypesenseFacetType::EntityReferenceObject->toGraphQlEnumValue();
        }
        else {
          $facetType = TypesenseFacetType::String->toGraphQlEnumValue();
        }

        $facets[$facetId] = [
          'id' => $enumKey,
          'type' => $facetType,
          'terms' => $terms,
        ];
      }
    }

    return array_values($facets);
  }

  /**
   * Processes a KeyValue facet.
   *
   * @param array $counts
   *   The facet counts from Typesense.
   *
   * @return array
   *   Array of facet terms with id and label from the facet value object.
   */
  private function processKeyValueFacet(array $counts): array {
    $terms = [];

    foreach ($counts as $count) {
      $value = $count['value'];
      $id = NULL;
      $label = NULL;

      // When facet_return_parent is used, Typesense returns the parent object
      // in a 'parent' property. For KeyValue facets, this contains the full object
      // with 'id' and 'label' properties.
      if (isset($count['parent']) && is_array($count['parent'])) {
        $parentData = $count['parent'];
        $id = $parentData['id'] ?? NULL;
        $label = $parentData['label'] ?? NULL;
      }

      // If we didn't get data from parent, try the value directly.
      if ($id === NULL || $label === NULL) {
        // For KeyValue facets, the value might be an object with 'id' and 'label' properties.
        if (is_array($value) && isset($value['id']) && isset($value['label'])) {
          $id = $value['id'];
          $label = $value['label'];
        }
        else {
          // Fallback: if the value is not in the expected format, use it as both id and label.
          $id = is_string($value) ? $value : (string) $value;
          $label = $id;
        }
      }

      $terms[] = [
        'id' => $id,
        'label' => $label,
        'count' => $count['count'],
        'terms' => [],
      ];
    }

    // Sort by label.
    usort($terms, function ($a, $b) {
      return strnatcasecmp($a['label'], $b['label']);
    });

    return $terms;
  }

  /**
   * Processes a normal string facet (non-hierarchical).
   *
   * @param array $counts
   *   The facet counts from Typesense.
   *
   * @return array
   *   Array of facet terms.
   */
  private function processStringFacet(array $counts): array {
    $terms = [];

    foreach ($counts as $count) {
      $id = $count['value'];
      // For string facets, the value itself is the label.
      $label = $id;

      $terms[] = [
        'id' => $id,
        'label' => $label,
        'count' => $count['count'],
        'terms' => [],
      ];
    }

    // Sort by label.
    usort($terms, function ($a, $b) {
      return strnatcasecmp($a['label'], $b['label']);
    });

    return $terms;
  }

  /**
   * Processes a hierarchical facet (taxonomy terms).
   *
   * @param array $counts
   *   The facet counts from Typesense.
   *
   * @return array
   *   Array of facet terms with hierarchy.
   */
  private function processHierarchicalFacet(array $counts): array {
    // Collect all nodes, track parent relationships.
    $nodes = [];
    // parentId => [ childId, ... ].
    $children = [];
    // Id => parentId.
    $parentMap = [];

    foreach ($counts as $count) {
      $id = $count['value'];

      // For taxonomy term facets, use the parent object data from Typesense.
      // The parent object contains: id, label, parent, weight, all_labels
      $parentData = $count['parent'] ?? [];

      $label = $parentData['label'] ?? $id;
      $parentId = $parentData['parent'] ?? NULL;
      $weight = $parentData['weight'] ?? NULL;
      $allLabels = $parentData['all_labels'] ?? NULL;

      // Stash the node with all taxonomy term properties.
      $nodes[$id] = [
        'id' => $id,
        'label' => $label,
        'count' => $count['count'],
        'weight' => $weight,
        'all_labels' => $allLabels,
      ];

      // Remember its parent.
      $parentMap[$id] = $parentId;

      // Index it under its parent (even if parent not seen yet).
      if ($parentId !== NULL) {
        $children[$parentId][] = $id;
      }
    }

    // Identify true roots: Those with no parent or parent outside this set.
    $roots = [];
    foreach (array_keys($nodes) as $id) {
      $pid = $parentMap[$id];
      if ($pid === NULL || !isset($nodes[$pid])) {
        $roots[] = $id;
      }
    }

    foreach (array_keys($children) as $pid) {
      if (!isset($nodes[$pid])) {
        // Skip orphans.
        continue;
      }

      // Sort children by weight (if available), then by label.
      usort(
        $children[$pid],
        function (string $a, string $b) use ($nodes) {
          $aWeight = $nodes[$a]['weight'] ?? NULL;
          $bWeight = $nodes[$b]['weight'] ?? NULL;

          // If both have weights, sort by weight.
          if ($aWeight !== NULL && $bWeight !== NULL) {
            if ($aWeight !== $bWeight) {
              return $aWeight <=> $bWeight;
            }
          }

          // Otherwise, sort by label.
          return strnatcasecmp($nodes[$a]['label'], $nodes[$b]['label']);
        }
      );
    }

    // Sort roots by weight (if available), then by label.
    usort($roots, function ($a, $b) use ($nodes) {
      $aWeight = $nodes[$a]['weight'] ?? NULL;
      $bWeight = $nodes[$b]['weight'] ?? NULL;

      // If both have weights, sort by weight.
      if ($aWeight !== NULL && $bWeight !== NULL) {
        if ($aWeight !== $bWeight) {
          return $aWeight <=> $bWeight;
        }
      }

      // Otherwise, sort by label.
      return strnatcasecmp($nodes[$a]['label'], $nodes[$b]['label']);
    });

    $tree = [];

    foreach ($roots as $rootId) {
      $tree[] = $this->buildTermTree((string) $rootId, $nodes, $children);
    }

    return $tree;
  }

  /**
   * Helper: build a single term (and its children) without using references.
   */
  private function buildTermTree(string $termId, array $nodes, array $children): array {
    $data = $nodes[$termId];
    $term = [
      'id' => $data['id'],
      'label' => $data['label'],
      'count' => $data['count'],
      'terms' => [],
    ];

    // Add taxonomy term specific properties if available.
    if (isset($data['weight'])) {
      $term['weight'] = $data['weight'];
    }
    if (isset($data['all_labels'])) {
      $term['all_labels'] = $data['all_labels'];
    }

    if (!empty($children[$termId])) {
      foreach ($children[$termId] as $childId) {
        $term['terms'][] = $this->buildTermTree((string) $childId, $nodes, $children);
      }
    }

    return $term;
  }

}
