<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\GraphQL\DataProducer;

use Drupal\typesense_graphql\Model\TypesenseMultiSearchResponse;
use Drupal\typesense_graphql\Model\TypesenseQuery;
use Drupal\typesense_graphql\Plugin\GraphQL\TypesenseDataProducerBase;
use Drupal\graphql\GraphQL\Execution\FieldContext;

/**
 * Performs a multiSearch Typesense query.
 *
 * @DataProducer(
 *   id = "typesense_graphql_perform_multi_search",
 *   name = "Perform Multi Search",
 *   description = @Translation("Performs a Typesense multiSearch query"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The response")
 *   ),
 *   consumes = {
 *     "query" = @ContextDefinition("any",
 *       label = @Translation("The TypesenseQuery"),
 *     ),
 *   }
 * )
 */
final class TypesensePerformMultiSearch extends TypesenseDataProducerBase {

  /**
   * Resolves the search results.
   *
   * @param \Drupal\typesense_graphql\Model\TypesenseQuery $query
   *   The query.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The GraphQL field context.
   *
   * @return \Drupal\typesense_graphql\Model\TypesenseMultiSearchResponse|null
   *   The Typesense response or null if an error occured.
   */
  protected function resolve(
    TypesenseQuery $query,
    FieldContext $field,
  ): TypesenseMultiSearchResponse|null {
    // Currently, this is the case when bad words are detected in the user's
    // search text. This marks the query as skippable, so we return a fake
    // '0 results' response.
    if ($query->isSkippable()) {
      return TypesenseMultiSearchResponse::createEmpty();
    }

    $collection = $query->getCollection();

    // We use this in the type resolvers for the hit/document interfaces to
    // resolve the correct type, because the hit itself doesn't contain
    // any reference to a collection.
    $field->setContextValue(
      'typesense_graphql_collection',
      $query->getAlias(),
    );

    $requestBody = $query->toQuery();

    try {
      $index = $query->getIndex();
      $client = $this->getClient($index, $field)->getClient();
      $rawResponse = $client->multiSearch->perform($requestBody);
      return new TypesenseMultiSearchResponse($rawResponse, $query);
    }
    catch (\Exception $e) {
      $field->mergeCacheMaxAge(60);

      $this->logger->error(
        "Error querying Typesense collection <strong>@collection</strong>.<br><strong>Error:</strong><br>@message<br><br><strong>Query:</strong><br><pre>{query}</pre>",
        [
          '@collection' => $collection,
          '@message' => $e->getMessage(),
          'query' => json_encode($query, JSON_PRETTY_PRINT),
        ]
      );
      return NULL;
    }
  }

}
