<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\search_api\data_type;

use Drupal\typesense_graphql\Plugin\search_api\EntityReferenceObjectDataTypeInterface;
use Drupal\search_api\DataType\DataTypePluginBase;
use Drupal\typesense_graphql\TypesenseSchemaAwareInterface;

/**
 * Provides a data type for "taxonomy_term" entity reference objects.
 *
 * @SearchApiDataType(
 *   id = "typesense_entity_reference_object_taxonomy_term[]",
 *   label = @Translation("Typesense: Entity Reference Object (Taxonomy Term)"),
 *   description = @Translation("A taxonomy term entity reference object."),
 * )
 */
class EntityReferenceObjectTaxonomyTerm extends DataTypePluginBase implements EntityReferenceObjectDataTypeInterface, TypesenseSchemaAwareInterface {

  /**
   * {@inheritdoc}
   */
  public function getEntityTypeId(): string {
    return 'taxonomy_term';
  }

  /**
   * {@inheritdoc}
   */
  public function alterSchemaField(array $field): array {
    // Entity reference objects should be stored as object[] in Typesense.
    $field['type'] = 'object[]';
    return $field;
  }

  /**
   * {@inheritdoc}
   */
  public function addAdditionalFields(array $field): array {
    $fieldName = $field['name'];
    $additionalFields = [];

    // We explicitly declare e.g. "brand.id" as field, because we need to
    // create facets on it.
    $additionalFields[] = [
      'name' => $fieldName . '.id',
      'type' => 'string[]',
      'facet' => TRUE,
      'optional' => TRUE,
      'index' => TRUE,
    ];

    // Explicitly declare e.g. "brand.label" as a field, because we use it
    // for fulltext search.
    $additionalFields[] = [
      'name' => $fieldName . '.label',
      'type' => 'string[]',
      'locale' => 'en',
      'facet' => FALSE,
      'optional' => TRUE,
      'index' => TRUE,
      'infix' => TRUE,
    ];

    // Explicitly declare the all_labels property as a field, because we
    // perform fulltext search on it.
    $additionalFields[] = [
      'name' => $fieldName . '.all_labels',
      'type' => 'string[]',
      'locale' => 'en',
      'facet' => FALSE,
      'optional' => TRUE,
      'index' => TRUE,
      'infix' => TRUE,
    ];

    return $additionalFields;
  }

}
