<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\search_api\data_type;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\rokka\RokkaServiceInterface;
use Drupal\search_api\DataType\DataTypePluginBase;
use Drupal\typesense_graphql\TypesenseSchemaAwareInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Typesense object data type to index a rokka image.
 *
 * The data type expects a value that is the ID of a media entity of bundle
 * "image".
 *
 * @SearchApiDataType(
 *   id = "typesense_rokka_image",
 *   label = @Translation("Typesense: Rokka Image"),
 *   description = @Translation("An object for a rokka image."),
 *   fallback_type = "string"
 * )
 */
class RokkaImage extends DataTypePluginBase implements TypesenseSchemaAwareInterface {

  /**
   * Constructs a new RokkaImageDataType object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\rokka\RokkaService $rokkaService
   *   The rokka service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RokkaServiceInterface $rokkaService,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('rokka.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getValue($value) {
    if (!$value) {
      return NULL;
    }

    /** @var \Drupal\media\Entity\Media|null $media */
    $media = $this->entityTypeManager->getStorage('media')->load($value);

    if (!$media) {
      return NULL;
    }

    $fid = $media->getSource()->getSourceFieldValue($media);
    if (!$fid) {
      return NULL;
    }

    $fileStorage = $this->entityTypeManager->getStorage('file');

    /** @var \Drupal\file\Entity\File|null $file */
    $file = $fileStorage->load($fid);

    if (!$file) {
      return NULL;
    }

    $uri = $file->uri?->value;

    if (!$uri) {
      return NULL;
    }

    $metadataEntities = array_values($this->rokkaService->loadRokkaMetadataByUri($uri));
    $metadata = $metadataEntities[0] ?? NULL;

    if (!$metadata) {
      return NULL;
    }

    $hash = $metadata->getHash();

    if (!$hash) {
      return NULL;
    }

    $imageField = $media->get('field_media_image');

    // Return an object so that the frontend can use existing components to
    // render the image.
    return [
      'hash' => $hash,
      'alt' => $imageField->alt,
      'title' => $imageField->title,
      'width' => (int) $imageField->width,
      'height' => (int) $imageField->height,
      'file_name' => $file->getFilename(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function alterSchemaField(array $field): array {
    // Rokka images should be stored as object in Typesense.
    $field['type'] = 'object';
    return $field;
  }

  /**
   * {@inheritdoc}
   */
  public function addAdditionalFields(array $field): array {
    return [];
  }

}
