<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\search_api\processor;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\typesense_graphql\Plugin\search_api\EntityReferenceObjectDataTypeInterface;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\taxonomy\TermInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Process "entity_reference_object" fields.
 *
 * The processor expects fields that contain one or more entity IDs of
 * entity reference fields. It then replaces these with objects containing
 * the id, label, parent and weight.
 *
 * The processor only handles fields whose data type plugin implements
 * the "EntityReferenceObjectDataTypeInterface".
 *
 * @SearchApiProcessor(
 *   id = "typesense_graphql_process_entity_reference_object",
 *   label = @Translation("Process Entity Reference Object"),
 *   description = @Translation("Process Entity Reference Object fields. This processor adds support for indexing entity reference fields as objects containing id, label, parent and weight. If you need hiearchy support, enable the related processor."),
 *   stages = {
 *     "preprocess_index" = -10,
 *   },
 * )
 */
class ProcessEntityReferenceObject extends ProcessorPluginBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|null
   */
  protected EntityTypeManagerInterface|null $entityTypeManager = NULL;

  /**
   * Cache built objects.
   *
   * A lot of entity reference objects are identical across jobs,
   * such as region, brand, etc., so we can cache them for the duration
   * of the batch processing.
   *
   * @var array<string, array>
   */
  protected array $objectCache = [];

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var static $processor */
    $processor = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $processor->setEntityTypeManager($container->get('entity_type.manager'));
    return $processor;
  }

  /**
   * Retrieves the entity type manager service.
   *
   * @return \Drupal\Core\Entity\EntityTypeManagerInterface
   *   The entity type manager service.
   */
  public function getEntityTypeManager() {
    // @phpstan-ignore-next-line This pattern to get a dependency is used by search_api.
    return $this->entityTypeManager ?: \Drupal::entityTypeManager();
  }

  /**
   * Sets the entity type manager service.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   *
   * @return $this
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function preprocessIndexItems(array $items) {
    foreach ($items as $item) {
      $fields = $item->getFields();
      $langcode = $item->getLanguage();

      foreach ($fields as $field) {
        $plugin = $field->getDataTypePlugin();

        if (!$plugin instanceof EntityReferenceObjectDataTypeInterface) {
          continue;
        }

        $entityType = $plugin->getEntityTypeId();
        $ids = $field->getValues();

        $newValues = [];

        if (is_array($ids)) {
          foreach ($ids as $id) {
            $newValues[] = $this->buildEntityReferenceObject(
              $entityType,
              (string) $id,
              $langcode
            );
          }
        }

        $newValues = array_filter($newValues);
        $field->setValues($newValues);
      }
    }
  }

  /**
   * Build the entity reference object.
   *
   * @param string $entityType
   *   The entity type.
   * @param string $id
   *   The entity ID.
   * @param string $langcode
   *   The langcode.
   *
   * @return array|null
   *   The entity reference object.
   */
  protected function buildEntityReferenceObject(string $entityType, string $id, string $langcode): array|null {
    $cacheKey = $entityType . $id . $langcode;
    $cached = $this->objectCache[$cacheKey] ?? NULL;

    if ($cached) {
      return $cached;
    }

    $storage = $this->getEntityTypeManager()->getStorage($entityType);
    $entity = $storage->load($id);

    if (!$entity) {
      return NULL;
    }

    $parentId = NULL;
    $weight = 0;

    // The label in the 'current' language.
    $label = $entity->label();

    // The labels in all languages.
    $allLabels = [];

    if ($entity instanceof TranslatableInterface) {
      if ($entity->hasTranslation($langcode)) {
        $label = $entity->getTranslation($langcode)->label();
      }

      $languages = $entity->getTranslationLanguages();
      foreach ($languages as $language) {
        $allLabels[] = (string) $entity->getTranslation($language->getId())->label();
      }
    }

    if ($entity instanceof TermInterface) {
      if ($entity->parent->target_id) {
        $parentId = (string) $entity->parent->target_id;
      }
      $weight = $entity->getWeight();
    }

    $object = [
      'id' => (string) $id,
      'label' => (string) $label,
      'all_labels' => array_unique($allLabels),
      'parent' => $parentId,
      'weight' => $weight,
    ];

    $this->objectCache[$cacheKey] = $object;

    return $object;
  }

}
