<?php

declare(strict_types=1);

namespace Drupal\typesense_graphql\Plugin\search_api\processor;

use Drupal\Core\Url;
use Drupal\search_api\LoggerTrait;
use Drupal\search_api\Processor\FieldsProcessorPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Process fields containing URLs.
 *
 * The processor will attempt to generate a valid URL from the value of fields
 * that contain link field values.
 *
 * @SearchApiProcessor(
 *   id = "typesense_graphql_valid_url",
 *   label = @Translation("Valid URL"),
 *   description = @Translation("Builds a valid URL from link fields."),
 *   stages = {
 *     "preprocess_index" = -15,
 *   }
 * )
 */
class ValidUrlProcessor extends FieldsProcessorPluginBase {

  use LoggerTrait;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var static $processor */
    $processor = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $processor->setLogger($container->get('logger.channel.search_api'));
    return $processor;
  }

  /**
   * {@inheritdoc}
   */
  protected function processFieldValue(&$value, $type) {
    $value = $this->parseUrl($value);
  }

  /**
   * Try to process the field value and convert it to a valid URL.
   *
   * @param mixed $value
   *   The possible value.
   *
   * @return string|null
   *   The valid URL or null.
   */
  private function parseUrl($value): string|null {
    if (!is_string($value)) {
      return NULL;
    }

    $trimmed = trim($value);

    if (empty($trimmed)) {
      return NULL;
    }

    try {
      $url = Url::fromUri($trimmed);
      return $url->toString();
    }
    catch (\Exception $e) {
      $this->logException($e);
    }

    return NULL;
  }

}
