<?php

/**
 * @file
 * Views field handler to render quick redeem/unredeem links in a View row.
 */

/**
 * Field handler: Redeemed total (computed).
 * Sums uc_product_redeem.qty for the (order_id, order_product_id) pair.
 */
class uc_product_redeem_views_field_redeemed_total extends views_handler_field {
  public function option_definition() {
    $o = parent::option_definition();
    $o['show_zero'] = array('default' => TRUE);
    return $o;
  }

  public function options_form(&$form, &$form_state) {
    parent::options_form($form, $form_state);
    $form['show_zero'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show 0 instead of blank when none redeemed'),
      '#default_value' => !empty($this->options['show_zero']),
    );
  }

  public function query() {
    $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    // We need these identifiers to compute the sum per row.
    // Manually add fields with explicit aliases to avoid ambiguity.
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id');
    $this->aliases['order_product_id'] = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id');
  }

  public function render($values) {
    $order_id = (int) $values->{$this->aliases['order_id']};
    $opid     = (int) $values->{$this->aliases['order_product_id']};
    if (!$order_id || !$opid) {
      return '';
    }
    $sum = uc_product_redeem_redeemed_qty($order_id, $opid);
    return ($sum === 0 && empty($this->options['show_zero'])) ? '' : (string) $sum;
  }
}

/**
 * Field handler: Remaining = ordered qty - redeemed sum.
 * Uses one cheap SUM() per row; fine for per-order displays.
 * Works with or without aggregation in the View.
 */
class uc_product_redeem_views_field_remaining extends views_handler_field {
  public function option_definition() {
    $options = parent::option_definition();
    $options['show_zero'] = array('default' => TRUE);
    return $options;
  }

  public function options_form(&$form, &$form_state) {
    parent::options_form($form, $form_state);
    $form['show_zero'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show 0 instead of blank when no redemptions exist'),
      '#default_value' => !empty($this->options['show_zero']),
    );
  }

  public function query() {
    $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    // We need order_id, order_product_id and ordered qty from uc_order_products.
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id');
    $this->aliases['order_product_id'] = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id');
    $this->aliases['qty'] = $this->query->add_field($table_alias, 'qty', 'ucpr_qty');
  }

  public function render($values) {
    $order_id = (int) $values->{$this->aliases['order_id']};
    $opid     = (int) $values->{$this->aliases['order_product_id']};

    // Prefer the view's row value for ordered qty (works with/without aggregation).
    $ordered = (int) $values->{$this->aliases['qty']};
    if ($ordered === NULL) {
      // Fallback: fetch ordered qty if not present (edge-case).
      $ordered = db_select('uc_order_products', 'op')
        ->fields('op', array('qty'))
        ->condition('order_id', $oid)
        ->condition('order_product_id', $opid)
        ->execute()
        ->fetchField();
    }
    $ordered = (int) $ordered;

    // Current redeemed sum.
    $redeemed = uc_product_redeem_redeemed_qty($order_id, $opid); 
    $remaining = max($ordered - $redeemed, 0);
    return ($remaining === 0 && empty($this->options['show_zero'])) ? '' : (string) $remaining;
  }
}

/**
 * First redeemed timestamp (MIN(created)) on order line.
 */
class uc_product_redeem_views_field_first_redeemed_ts extends views_handler_field {
  public function query() {
    $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id');
    $this->aliases['order_product_id'] = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id');
  }
  public function render($values) {
    $order_id = (int) $values->{$this->aliases['order_id']};
    $opid     = (int) $values->{$this->aliases['order_product_id']};
    if (!$order_id || !$opid) return '';
    $min = db_select('uc_product_redeem', 'upr')
      ->condition('order_id', $order_id)
      ->condition('order_product_id', $opid)
      ->condition('qty', 0, '>');
      $min->addExpression('MIN(created)', 'min_created');
      $ts = $min->execute()->fetchField();
    return $ts ? format_date((int)$ts, 'short') : '';
  }
}

/**
 * Last redeemed timestamp (MAX(created)) on order line.
 */
class uc_product_redeem_views_field_last_redeemed_ts extends views_handler_field {
  public function query() {
    $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id');
    $this->aliases['order_product_id'] = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id');
  }
  public function render($values) {
    $order_id = (int) $values->{$this->aliases['order_id']};
    $opid     = (int) $values->{$this->aliases['order_product_id']};
    if (!$order_id || !$opid) return '';
    $max = db_select('uc_product_redeem', 'upr')
      ->condition('order_id', $order_id)
      ->condition('order_product_id', $opid)
      ->condition('qty', 0, '>');
      $max->addExpression('MAX(created)', 'max_created');
    $ts = $max->execute()->fetchField();
    return $ts ? format_date((int)$ts, 'short') : '';
  }
}

/**
 * Last redemption note on order line.
 */
class uc_product_redeem_views_field_last_note extends views_handler_field {
  public function query() {
    $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id');
    $this->aliases['order_product_id'] = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id');
  }
  public function render($values) {
    $order_id = (int) $values->{$this->aliases['order_id']};
    $opid     = (int) $values->{$this->aliases['order_product_id']};
    if (!$order_id || !$opid) return '';
    $query = db_select('uc_product_redeem', 'upr')
      ->fields('upr', array('note'))
      ->condition('order_id', $order_id)
      ->condition('order_product_id', $opid)
      ->condition('note', '', '!=')
      ->orderBy('created', 'DESC')
      ->range(0, 1);
    $note = $query->execute()->fetchField();  
    return $note ? check_plain($note) : '';
  }
}

/**
 * Last redeemed by (username) on order line.
 */
class uc_product_redeem_views_field_last_by extends views_handler_field {
  public function query() {
    $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id');
    $this->aliases['order_product_id'] = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id');
  }
  public function render($values) {
    $order_id = (int) $values->{$this->aliases['order_id']};
    $opid     = (int) $values->{$this->aliases['order_product_id']};
    if (!$order_id || !$opid) return '';
    $query = db_select('uc_product_redeem', 'upr')
      ->fields('upr', array('redeemed_by_name'))
      ->condition('order_id', $order_id)
      ->condition('order_product_id', $opid)
      ->condition('qty', 0, '>')
      ->orderBy('created', 'DESC')
      ->range(0, 1);
    $by = $query->execute()->fetchField();
    return $by ? check_plain($by) : '';
  }
}

/**
 * Per-line "Redeem 1" / "Unredeem 1" actions.
 * Base table: uc_order_products
 */
class uc_product_redeem_views_field_actions extends views_handler_field {
  function option_definition() {
    $options = parent::option_definition();
    $options['show_redeem']   = array('default' => TRUE);
    $options['show_unredeem'] = array('default' => TRUE);
    return $options;
  }

  function options_form(&$form, &$form_state) {
    parent::options_form($form, $form_state);
    $form['show_redeem'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show “Redeem 1”'),
      '#default_value' => !empty($this->options['show_redeem']),
    );
    $form['show_unredeem'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show “Unredeem 1”'),
      '#default_value' => !empty($this->options['show_unredeem']),
    );
  }

  function query() {
    $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id');
    $this->aliases['order_product_id'] = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id');
  }

  function render($values) {
    // Extract identifiers.
    $order_id = (int) $values->{$this->aliases['order_id']};
    $order_product_id     = (int) $values->{$this->aliases['order_product_id']};

    if (!$order_id || !$order_product_id) {
      return '';
    }
    $order_item = uc_product_redeem_order_item($order_id, $order_product_id); 

    // Compute token once per row.
    if (!function_exists('uc_product_redeem_token')) {
      return '';
    }
    $token = uc_product_redeem_token($order_id, $order_product_id);

    $links = array();

    // We can show links unconditionally; apply() clamps/validates.
    // (Optionally, you could preload remaining/redeemed and conditionally hide.)
    if (!empty($this->options['show_redeem'])) {
      if ($order_item->can_redeem){        
        $q = array(
          'opid' => $order_product_id, 
          'qty' => 1, 
          'token' => $token
        );
        if (isset($_GET['ak'])) { 
          $q['ak'] = $_GET['ak']; 
        }
        $links[] = l(t('+1'), "product-redeem/$order_id/redeem", array(
          'query' => $q,
          'attributes' => array('class' => array('form-submit', 'button', 'button--small', 'btn-redeem', 'btn-redeem-one')),
        ));
      }
    }

    if (!empty($this->options['show_unredeem'])) {
      if ($order_item->can_unredeem){ 
        $q = array(
          'opid' => $order_product_id, 
          'qty' => -1, 
          'token' => $token
        );
        if (isset($_GET['ak'])) { $q['ak'] = $_GET['ak']; }
        $links[] = l(t('-1'), "product-redeem/$order_id/redeem", array(
          'query' => $q,
          'attributes' => array('class' => array('form-submit', 'button', 'button--small', 'button--alt', 'btn-unredeem', 'btn-unredeem-one')),
        ));
      }
    }

    return empty($links) ? '' : '<div class="ucpr-quick-links">'. implode(' ', $links) .'</div>';
  }
}

/**
 * Per-line "Redeem all" / "Unredeem all" actions.
 * Base table: uc_order_products
 */
class uc_product_redeem_views_field_actions_all extends views_handler_field {
  public function option_definition() {
    $o = parent::option_definition();
    $o['show_redeem_all']   = array('default' => TRUE);
    $o['show_unredeem_all'] = array('default' => TRUE);
    return $o;
  }

  public function options_form(&$form, &$form_state) {
    parent::options_form($form, $form_state);
    $form['show_redeem_all'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show "Redeem all"'),
      '#default_value' => !empty($this->options['show_redeem_all']),
    );
    $form['show_unredeem_all'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show "Unredeem all"'),
      '#default_value' => !empty($this->options['show_unredeem_all']),
    );
  }

  public function query() {
    // Use the alias Views already assigned to our base table.
    $table_alias = $this->table_alias;
    if (!$table_alias) {
      $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    }

    // Add the identifiers with explicit aliases to avoid ambiguity.
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id_all');
    $this->aliases['order_product_id']     = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id_all');
    $this->aliases['qty']      = $this->query->add_field($table_alias, 'qty', 'ucpr_ordered_qty_all');
  }

  public function render($values) {
    $order_id   = (int) $values->{$this->aliases['order_id']};
    $order_product_id       = (int) $values->{$this->aliases['order_product_id']};
    $ordered    = (int) $values->{$this->aliases['qty']};

    if (!$order_id || !$order_product_id) {
      return '';
    }

    // Compute redeemed/remaining using module helper.
    $redeemed = (int) uc_product_redeem_redeemed_qty($order_id, $order_product_id);
    $remaining = max($ordered - $redeemed, 0);

    $token = uc_product_redeem_token($order_id, $order_product_id);
    $links = array();

    if (!empty($this->options['show_redeem_all']) && $remaining > 0) {
      $q = array(
        'opid' => $order_product_id, 
        'qty' => $remaining, 
        'token' => $token
      );
      if (isset($_GET['ak'])) $q['ak'] = $_GET['ak'];
      $links[] = l(t('Redeem all', array('@n' => $remaining)), "product-redeem/$order_id/redeem", array(
        'query' => $q,
        'attributes' => array('class' => array('form-submit','button', 'btn-redeem')),
      ));
    }

    if (!empty($this->options['show_unredeem_all']) && $redeemed > 0) {
      $q = array(
        'opid' => $order_product_id, 
        'qty' => -$redeemed, 
        'token' => $token
      );
      if (isset($_GET['ak'])) $q['ak'] = $_GET['ak'];
      $links[] = l(t('Unredeem all', array('@n' => $redeemed)), "product-redeem/$order_id/redeem", array(
        'query' => $q,
        'attributes' => array('class' => array('form-submit','button', 'button--alt', 'btn-unredeem')),
      ));
    }

    return $links ? '<div class="ucpr-quick-links ucpr-actions-all">' . implode(' ', $links) . '</div>' : '';
  }
}

/**
 * Per-line "Redeem X" / "Unredeem X" actions (with qty input).
 * Base table: uc_order_products
 */
class uc_product_redeem_views_field_actions_x extends views_handler_field {
  public function option_definition() {
    $o = parent::option_definition();
    $o['show_redeem_x']   = array('default' => TRUE);
    $o['show_unredeem_x'] = array('default' => TRUE);
    $o['default_qty']     = array('default' => 1);
    return $o;
  }

  public function options_form(&$form, &$form_state) {
    parent::options_form($form, $form_state);
    $form['show_redeem_x'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show "Redeem X"'),
      '#default_value' => !empty($this->options['show_redeem_x']),
    );
    $form['show_unredeem_x'] = array(
      '#type' => 'checkbox',
      '#title' => t('Show "Unredeem X"'),
      '#default_value' => !empty($this->options['show_unredeem_x']),
    );
    $form['default_qty'] = array(
      '#type' => 'textfield',
      '#title' => t('Default quantity'),
      '#size' => 4,
      '#default_value' => isset($this->options['default_qty']) ? (int) $this->options['default_qty'] : 1,
    );
  }

  public function query() {
    $table_alias = $this->table_alias;
    if (!$table_alias) {
      $table_alias = $this->query->ensure_table($this->table, $this->relationship);
    }
    $this->aliases['order_id'] = $this->query->add_field($table_alias, 'order_id', 'ucpr_order_id_x');
    $this->aliases['order_product_id']     = $this->query->add_field($table_alias, 'order_product_id', 'ucpr_order_product_id_x');
    $this->aliases['qty']      = $this->query->add_field($table_alias, 'qty', 'ucpr_ordered_qty_x');
  }

  public function render($values) {
    $order_id   = (int) $values->{$this->aliases['order_id']};
    $order_product_id       = (int) $values->{$this->aliases['order_product_id']};
    $ordered    = (int) $values->{$this->aliases['qty']};

    if (!$order_id || !$order_product_id) {
      return '';
    }

    $redeemed  = (int) uc_product_redeem_redeemed_qty($order_id, $order_product_id);
    $remaining = max($ordered - $redeemed, 0);
    $token     = uc_product_redeem_token($order_id, $order_product_id);

    $default_qty = max(1, (int) $this->options['default_qty']);

    $out = array();

    if (!empty($this->options['show_redeem_x']) && $remaining > 0) {
      // Redeem X mini-form (GET submit to canonical route).
      $q_ak = isset($_GET['ak']) ? '<input type="hidden" name="ak" value="' . check_plain($_GET['ak']) . '" />' : '';
      $out[] =
        '<form class="ucpr-x-form ucpr-x-form--redeem" method="get" action="' . check_plain(url("product-redeem/$order_id/redeem")) . '">' .
          '<input type="hidden" name="opid" value="' . (int) $order_product_id . '" />' .
          '<input type="hidden" name="token" value="' . check_plain($token) . '" />' .
          $q_ak .
          '<input class="ucpr-x-input" type="number" name="qty" min="1" max="' . $remaining . '" value="' . $default_qty . '" /> ' .
          '<button class="button form-submit btn-redeem" type="submit">' . t('Redeem') . '</button>' .
          //'<span class="ucpr-x-hint">' . t('Max: @n', array('@n' => $remaining)) . '</span>' .
        '</form>';
    }

    if (!empty($this->options['show_unredeem_x']) && $redeemed > 0) {
      // Unredeem X mini-form (negative qty).
      $q_ak = isset($_GET['ak']) ? '<input type="hidden" name="ak" value="' . check_plain($_GET['ak']) . '" />' : '';
      $out[] =
        '<form class="ucpr-x-form ucpr-x-form--unredeem" method="get" action="' . check_plain(url("product-redeem/$order_id/redeem")) . '" onsubmit="this.qty.value = -Math.abs(this.qty.value);">' .
          '<input type="hidden" name="opid" value="' . (int) $order_product_id . '" />' .
          '<input type="hidden" name="token" value="' . check_plain($token) . '" />' .
          $q_ak .
          '<input class="ucpr-x-input" type="number" name="qty" min="1" max="' . $redeemed . '" value="' . min($default_qty, $redeemed) . '" /> ' .
          '<button class="form-submit button button--alt btn-unredeem" type="submit">' . t('Unredeem') . '</button>' .
          //'<span class="ucpr-x-hint">' . t('Max: @n', array('@n' => $redeemed)) . '</span>' .
        '</form>';
    }

    return $out ? '<div class="ucpr-actions-x">' . implode('', $out) . '</div>' : '';
  }
}