<?php

/**
 * @file
 * Page callbacks and forms for Ubercart Product Redeem.
 */

/**
 * Decode your obfuscated order id from ?o=XXXX<oid*2>YYYY.
 */
function _ucpr_decode_obfuscated_order_id($o) {
  if (!is_string($o) || strlen($o) < 9) {
    return 0;
  }
  $middle = substr($o, 4, strlen($o) - 8); // strip first & last 4 chars
  if (!ctype_digit($middle)) {
    return 0;
  }
  return (int) ((int) $middle / 2);
}

/**
 * Legacy entry point compatible with old QR links.
 * Example: /ticketing/redeem?o=<rand><oid*2><rand>&c=<created>[&a=ri|ui|ro&li=<opid>&qty=<n>]
 */
function uc_product_redeem_legacy_entry() {
  $ak = isset($_GET['k']) ? $_GET['k'] : NULL;
  $o  = isset($_GET['o'])  ? $_GET['o']  : '';
  $oid = _ucpr_decode_obfuscated_order_id($o);
  if ($oid <= 0) {
    drupal_set_message(t('Invalid redemption link.'), 'error');
    return drupal_goto('product-redeem');
  }

  $order = uc_order_load($oid);
  if (!$order) {
    drupal_set_message(t('Order not found.'), 'error');
    return drupal_goto('product-redeem');
  }

  // Access check consistent with canonical page.
  if (!uc_product_redeem_access('redeem', $order)) {
    return MENU_ACCESS_DENIED;
  }

  // Back-compat actions: a=ri (redeem item), a=ui (unredeem item), a=ro (redeem order).
  if (isset($_GET['a'])) {
    $a = $_GET['a'];
    switch ($a) {
      case 'ri':
      case 'ui':
        $opid = isset($_GET['li']) ? (int) $_GET['li'] : 0;
        $qty  = isset($_GET['qty']) ? (int) $_GET['qty'] : 0;
        if ($opid > 0 && $qty > 0) {
          if ($a === 'ui') $qty = -$qty; // negative to unredeem
          $ok = uc_product_redeem_apply($order, $opid, $qty, 'legacy-link');
          drupal_set_message($ok ? t('Action completed.') : t('No action performed.'), $ok ? 'status' : 'warning');
        }
        break;

      case 'ro':
        // Redeem all remaining.
        $items = uc_product_redeem_order_items($order->order_id);
        $did = 0;
        foreach ($items as $it) {
          if ($it->remaining > 0) {
            if (uc_product_redeem_apply($order, $it->order_product_id, $it->remaining, 'legacy-bulk')) {
              $did += $it->remaining;
            }
          }
        }
        if ($did > 0) {
          drupal_set_message(t('Redeemed @n item(s) in total.', array('@n' => $did)));
        } else {
          drupal_set_message(t('Nothing to redeem.'), 'warning');
        }
        break;
    }
  }

  // Redirect to canonical page, preserving ?ak if present.
  $query = array();
  if ($ak !== NULL) $query['k'] = $ak;
  return drupal_goto('product-redeem/' . $order->order_id . '/redeem', array('query' => $query));
}

/**
 * Finder: enter order id (and optional email) to jump to page.
 */
function uc_product_redeem_finder_form($form, &$form_state) {
  $form['intro'] = array('#markup' => '<p>' . t('Enter an Ubercart order ID to manage redemptions.') . '</p>');

  $form['order_id'] = array(
    '#type' => 'textfield',
    '#title' => t('Order ID'),
    '#required' => TRUE,
    '#size' => 12,
  );
  $form['email'] = array(
    '#type' => 'textfield',
    '#title' => t('Customer email (optional)'),
    '#size' => 30,
    '#description' => t('If provided, must match the order.'),
  );

  $form['actions'] = array('#type' => 'actions');
  $form['actions']['go'] = array('#type' => 'submit', '#value' => t('Find order'));

  return $form;
}

function uc_product_redeem_finder_form_submit($form, &$form_state) {
  $oid = (int) $form_state['values']['order_id'];
  $order = $oid > 0 ? uc_order_load($oid) : FALSE;
  if (!$order) {
    form_set_error('order_id', t('Order not found.'));
    return;
  }
  $email = trim($form_state['values']['email']);
  if ($email !== '' && strcasecmp($email, $order->primary_email) !== 0) {
    form_set_error('email', t('Email does not match the order.'));
    return;
  }
  $dest = 'product-redeem/' . $oid . '/redeem';
  if (isset($_GET['k'])) {
    $dest .= '?ak=' . urlencode($_GET['k']);
  }
  $form_state['redirect'] = $dest;
}

/**
 * Controller for the redemption page.
 */
function uc_product_redeem_order_page($order) {
  // Quick-link actions.
  if (isset($_GET['opid'], $_GET['qty'], $_GET['token'])) {
    $opid = (int) $_GET['opid'];
    $qty = (int) $_GET['qty']; // positive for redeem, negative for unredeem
    if (hash_equals(uc_product_redeem_token($order->order_id, $opid), $_GET['token'])) {
      $ok = uc_product_redeem_apply($order, $opid, $qty, ($qty >= 0 ? 'quick-redeem' : 'quick-unredeem'));
      drupal_set_message($ok
        ? t('Action completed.')
        : t('No action performed.'), $ok ? 'status' : 'warning');
    }
    else {
      drupal_set_message(t('Invalid token.'), 'error');
    }
    $url = 'product-redeem/' . $order->order_id . '/redeem';
    if (isset($_GET['k'])) {
      $url .= '?ak=' . urlencode($_GET['k']);
    }
    drupal_goto($url);
  }

  // Accept either an order object or an order id (defensive).
  if (is_numeric($order)) {
    $order = uc_order_load((int) $order);
  }
  if (!$order || !is_object($order)) {
    drupal_set_message(t('Order not found.'), 'error');
    return array('#markup' => '');
  }
  
  $items = uc_product_redeem_order_items($order->order_id);
  $viewable_count = count($items);

  $intro_default = variable_get('uc_product_redeem_page_intro', array('value' => '', 'format' => filter_default_format()));
  $settings = array(
    'intro' => $intro_default['value'],
    'scan_mode' => (bool) variable_get('uc_product_redeem_scan_mode', FALSE),
    'allow_partial' => (bool) variable_get('uc_product_redeem_allow_partial', TRUE),
    'filter_to_now' => (bool) variable_get('uc_product_redeem_filter_to_now', FALSE),
    'allow_notes' => (bool) variable_get('uc_product_redeem_allow_notes', TRUE),
    'show_comments' => (bool) variable_get('uc_product_redeem_show_comments', TRUE),
    'comments_collapsed' => (bool) variable_get('uc_product_redeem_comments_collapsed', TRUE),
    'show_prices' => (bool) variable_get('uc_product_redeem_show_prices', FALSE),
    'show_quick_links'  => (bool) variable_get('uc_product_redeem_show_quick_links', TRUE),
    'show_redeemed'  => (bool) variable_get('uc_product_redeem_show_redeemed', TRUE),
    'show_remaining' => (bool) variable_get('uc_product_redeem_show_remaining', TRUE),    
    'embed_view' => (bool) variable_get('uc_product_redeem_use_views_table', FALSE),    
  );

  $build = array();

  // Let the user know no products are available to be redeemed/unredeemed
  if ($settings['filter_to_now'] && $viewable_count == 0 && !$settings['embed_view']){
    drupal_set_message(t('There are no items valid for today’s date.'), 'warning');
  }

  // Intro text moved to template
  //if (!empty($settings['intro'])) {
  //  $build['intro'] = array('#markup' => $settings['intro'], '#weight' => -10);
  //}

  $build['page'] = array(
    '#theme' => 'uc_product_redeem_page',
    '#order' => $order,
    '#items' => $items,
    '#settings' => $settings,
    '#viewable_count' => $viewable_count,
    '#weight' => 0,
  );

  // Either embed a Views display (if configured), or use the built-in table.
  $use_views = module_exists('views') && variable_get('uc_product_redeem_use_views_table', FALSE);
  $view_spec = variable_get('uc_product_redeem_views_table_id', ''); // e.g. "ucpr_order_lines:embed"

  if ($use_views && preg_match('/^([a-z0-9_]+):([a-z0-9_]+)$/', $view_spec, $m)) {
    $view_name   = $m[1];
    $display_id  = $m[2];

    // Pass order_id as a contextual filter to the View.
    $rendered = views_embed_view($view_name, $display_id, $order->order_id);
    $build['items_table'] = array(
      '#type' => 'markup',
      '#markup' => $rendered,
      '#weight' => 10,
    );
  }
  else {
    // Fallback: native table (current stable UI).
    // Items table (NO outer form → prevents nested forms).
    $build['items_table'] = uc_product_redeem_table_build($order, $items, $settings);
  }

  // Optional: Order comments panel (collapsible).
  if ($settings['show_comments']) {
    $build['order_comments'] = array(
      '#markup' => uc_product_redeem_order_comments_markup($order->order_id, !$settings['comments_collapsed']),
      '#weight' => 50,
    );
  }

  // Notes panel (collapsible, its own form, writes one order comment).
  if ($settings['allow_notes']) {
    $build['note_form'] = drupal_get_form('uc_product_redeem_note_form', $order->order_id);
    $build['note_form']['#weight'] = 55;
  }

  if ($viewable_count > 0) {
    //$build['form'] = drupal_get_form('uc_product_redeem_form', $order, $items);
    $build['form_all'] = drupal_get_form('uc_product_redeem_all_form', $order, $items);
    $build['form_all']['#weight'] = 60;
  }

  return $build;
}

/**
 * Build the items table with per-row mini-forms (no nesting).
 */
function uc_product_redeem_table_build($order, $items, $settings = []) {
  // Resolve flags (prefer $settings, fall back to variables).
  $show_prices    = isset($settings['show_prices'])    ? (bool) $settings['show_prices']    : (bool) variable_get('uc_product_redeem_show_prices', FALSE);
  $show_redeemed  = isset($settings['show_redeemed'])  ? (bool) $settings['show_redeemed']  : (bool) variable_get('uc_product_redeem_show_redeemed', TRUE);
  $show_remaining = isset($settings['show_remaining']) ? (bool) $settings['show_remaining'] : (bool) variable_get('uc_product_redeem_show_remaining', TRUE);

  // Header: build in the same order we'll render cells.
  $header = array();
  $header[] = array('data' => t('Item'));

  if ($show_prices){ 
    $header[] = array('data' => t('Price')); 
  }
  $header[] = array('data' => t('Qty'));
  
  if ($show_redeemed){ 
    $header[] = array('data' => t('Redeemed')); 
  }
  if ($show_remaining){ 
    $header[] = array('data' => t('Remaining')); 
  }
  $header[] = array('data' => t('Action'));

  $rows = array();

  foreach ($items as $it) {
    // Attributes (Seating/Seat friendly).
    $attr_html = '';
    $data = @unserialize($it->data);
    if (is_array($data) && isset($data['attributes']) && is_array($data['attributes'])) {
      foreach ($data['attributes'] as $label => $values) {
        if (!is_array($values)) continue;
        if ($label == 'Seating' || $label == 'Seat') {
          $attr_html .= '<div><strong>' . check_plain($label) . ':</strong> ';
          $parts_out = array();
          foreach ($values as $opt) {
            $parts = explode('|', trim($opt));
            if (isset($parts[1], $parts[2])) {
              $parts_out[] = '<b>' . check_plain($parts[1] . $parts[2]) . '</b>';
            } else {
              $parts_out[] = check_plain($opt);
            }
          }
          $attr_html .= implode(', ', $parts_out) . '</div>';
        }
        else {
          $attr_html .= '<div><strong>' . check_plain($label) . ':</strong> ' . check_plain(implode(', ', $values)) . '</div>';
        }
      }
    }

    $item_html = '<div class="ucpr-item-title">' . check_plain($it->title) . ($it->model ? ' <small>(' . check_plain($it->model) . ')</small>' : '') . '</div>' . $attr_html;

    $cells = array();
    $cells[] = array('data' => $item_html);
    if ($show_prices) { 
      $cells[] = uc_currency_format($it->price); 
    }
    $cells[] = $it->qty;
    if ($show_redeemed)  { $cells[] = $it->redeemed; }
    if ($show_remaining) { $cells[] = $it->remaining; }       
    //$cells[] = $it->redeemed;
    //$cells[] = $it->remaining;

    // Compact action cell: single qty + action select + one button.
    //$inline_form = drupal_get_form('uc_product_redeem_inline_row_form', $order, $it, $settings);
    // Use a unique form_id per row:
    $inline_form = drupal_get_form('uc_product_redeem_inline_row_form__' . $it->order_product_id, $order, $it, $settings);
    $inline_markup = drupal_render($inline_form);
    $cells[] = array('data' => $inline_markup, 'class' => array('ucpr-actions-cell'));

    $rows[] = array(
      'data' => $cells,
      'class' => array($it->remaining ? 'available' : 'redeemed'),
    );
  }

  return array(
    '#type' => 'markup',
    '#markup' => theme('table', array('header' => $header, 'rows' => $rows, 'attributes' => array('class' => array('ucpr-table')))),
    '#weight' => 10,
  );
}

/**
 * Per-row mini form:
 * - If partial enabled: Qty + (Redeem | Unredeem) buttons.
 * - If partial disabled: (Redeem all | Unredeem all) buttons, no qty.
 * - Optional quick links per settings.
 */
function uc_product_redeem_inline_row_form($form, &$form_state, $order, $it, $settings) {
  $form['order_id'] = array('#type' => 'hidden', '#value' => $order->order_id);
  $form['opid'] = array('#type' => 'hidden', '#value' => $it->order_product_id);

  $allow_partial = !empty($settings['allow_partial']);

  $form['ucpr_item_actions'] = array(
    '#type' => 'container',
    '#attributes' => array(
      'class' => ['ucpr-actions']
    ),
  );

  if ($allow_partial) {
    $max_for_action = max($it->remaining, $it->redeemed, 1);
    $default_qty = 1;

    $form['ucpr_item_actions']['qty'] = array(
      '#type' => 'textfield',
      '#title' => t('Qty'),
      '#title_display' => 'invisible',
      '#size' => 4,
      '#min' => 1,
      '#max' => $max_for_action,
      '#default_value' => $default_qty,
      '#attributes' => array('class' => array('ucpr-qty')),
    );
    if ($it->qty == 1) {
      // Don't show the input field if its only 1 item
      $form['ucpr_item_actions']['qty']['#attributes']['class'][] = 'hidden';
    }  

    // Redeem.
    if ($it->remaining > 0) {
      $form['ucpr_item_actions']['btn_redeem'] = array(
        '#type' => 'submit',
        '#value' => t('Redeem'),
        '#submit' => array('uc_product_redeem_inline_row_submit_redeem'),
        '#validate' => array('uc_product_redeem_inline_row_validate_redeem'),
        //'#limit_validation_errors' => array(array('qty')),
        '#attributes' => array(
          'class' => ['btn-redeem'],
          'data-scan-submit' => $settings['scan_mode'] ? '1' : '0',
        ),
      );
    }

    // Unredeem.
    if ($it->redeemed > 0) {
      $form['ucpr_item_actions']['btn_unredeem'] = array(
        '#type' => 'submit',
        '#value' => t('Unredeem'),
        '#submit' => array('uc_product_redeem_inline_row_submit_unredeem'),
        '#validate' => array('uc_product_redeem_inline_row_validate_unredeem'),
        '#attributes' => array(
          'class' => ['btn-unredeem'],
        ),
      );
    }
  }
  else {
    // Partial OFF: no qty.   
    if ($it->remaining > 0) {
      $form['ucpr_item_actions']['btn_redeem_all'] = array(
        '#type' => 'submit',
        '#value' => t('Redeem all'),
        '#submit' => array('uc_product_redeem_inline_row_submit_redeem_all'),
        '#attributes' => array(
          'class' => ['btn-redeem'],
        ),        
      );
    }
    if ($it->redeemed > 0) {
      $form['ucpr_item_actions']['btn_unredeem_all'] = array(
        '#type' => 'submit',
        '#value' => t('Unredeem all'),
        '#submit' => array('uc_product_redeem_inline_row_submit_unredeem_all'),
        '#attributes' => array(
          'class' => ['btn-unredeem'],
        ),
      );
    }
  }

  // Quick links (optional).
  if (!empty($settings['show_quick_links'])) {
    $q = array('opid' => $it->order_product_id, 'token' => $it->token);
    if (isset($_GET['k'])) $q['k'] = $_GET['k'];

    $links = array();
    if ($it->can_redeem) {
      $q1 = $q; 
      $q1['qty'] = 1;
      $links[] = l(t('+1'), 'product-redeem/' . $order->order_id . '/redeem', array('query' => $q1, 'attributes' => array('class' => array('button', 'button--small', 'form-submit', 'btn-redeem', 'btn-redeem-one'))));
    }
    if ($it->can_unredeem) {
      $q2 = $q; 
      $q2['qty'] = -1;
      $links[] = l(t('-1'), 'product-redeem/' . $order->order_id . '/redeem', array('query' => $q2, 'attributes' => array('class' => array('button', 'button--small', 'button--alt', 'form-submit', 'btn-unredeem', 'btn-unredeem-one'))));
    }

    if (!empty($links)) {
      $form['ucpr_item_actions']['quick'] = array('#type' => 'markup', '#markup' => '<div class="ucpr-quick-links">' . implode(' ', $links) . '</div>');
    }
  }

  return $form;
}
/** Validate: partial ON, Redeem */
function uc_product_redeem_inline_row_validate_redeem($form, &$form_state) {
  //watchdog('uc_product_redeem', 'Debugging inline-redeem-validate form_state: <pre>@form</pre>', array('@form' => print_r($form_state['values'], true), ), WATCHDOG_NOTICE);
  $order_id = $form_state['values']['order_id'];
  $opid     = $form_state['values']['opid'];
  $qty = (int) $form_state['values']['qty'];

  if (! $opid) {
    form_set_error('', t('Line item not found.'));
    return;
  }

  if ($qty <= 0) {
    form_set_error('qty', t('Quantity must be at least 1.'));
    return;
  }
  
  $items    = uc_product_redeem_order_items($order_id);
  foreach ($items as $it) {
    if ($it->order_product_id == $opid) {
      if ($it->remaining <= 0) {
        form_set_error('qty', t('Nothing left to redeem.'));
        return;
      } elseif ($qty > $it->remaining) {
        form_set_error('qty', t('You can redeem at most @n.', array('@n' => $it->remaining)));
        return;
      }
    }
  }

}

/** Validate: partial ON, Unredeem */
function uc_product_redeem_inline_row_validate_unredeem($form, &$form_state) {
  $order_id = $form_state['values']['order_id'];
  $opid     = $form_state['values']['opid'];
  $qty = (int) $form_state['values']['qty'];

  if (! $opid) {
    form_set_error('', t('Line item not found.'));
    return;
  }  

  if ($qty <= 0) {
    form_set_error('qty', t('Quantity must be at least 1.'));
    return;
  }

  $items    = uc_product_redeem_order_items($order_id);
  foreach ($items as $it) {
    if ($it->order_product_id == $opid) {
      if ($it->redeemed <= 0) {
        form_set_error('qty', t('Nothing to unredeem.'));
        return;
      } elseif ($qty > $it->redeemed) {
        form_set_error('qty', t('You can unredeem at most @n.', array('@n' => $it->redeemed)));
        return;
      }
    }
  }
}

/** Submit: partial ON, Redeem */
function uc_product_redeem_inline_row_submit_redeem($form, &$form_state) {
  //watchdog('uc_product_redeem', 'Debugging inline-redeem form_state: <pre>@form</pre>', array('@form' => print_r($form_state['values'], true), ), WATCHDOG_NOTICE);
  if (isset($form_state['values']['order_id'])){
    $order = uc_order_load($form_state['values']['order_id']);
    $opid  = (int) $form_state['values']['opid'];
  }else if (isset($form_state['complete form'])){
    // IF validation errors is set on the element use complete form because `values` will not contain what we want
    $order = uc_order_load($form_state['complete form']['order_id']['#value']);
    $opid = (int) $form_state['complete form']['opid']['#value'];
  }

  if (!$order || !$opid){
    return drupal_set_message(t('Order ID not valid', array()));
  }

  $qty  = (int) $form_state['values']['qty'];
  if (uc_product_redeem_apply($order, $opid, $qty, 'inline')) {
    drupal_set_message(t('Redeemed @n item(s).', array('@n' => $qty)));
  }
  _uc_product_redeem_redirect_after($order->order_id);
}

/** Submit: partial ON, Unredeem */
function uc_product_redeem_inline_row_submit_unredeem($form, &$form_state) {
  if (isset($form_state['values']['order_id'])){
    $order = uc_order_load($form_state['values']['order_id']);
    $opid  = (int) $form_state['values']['opid'];
  }else if (isset($form_state['complete form'])){
    // IF validation errors is set on the element use complete form because `values` will not contain what we want
    $order = uc_order_load($form_state['complete form']['order_id']['#value']);
    $opid = (int) $form_state['complete form']['opid']['#value'];
  }

  if (!$order || !$opid){
    return drupal_set_message(t('Order ID or Order product ID not valid', array()));
  }

  $qty   = (int) $form_state['values']['qty'];
  if (uc_product_redeem_apply($order, $opid, -$qty, 'inline')) {
    drupal_set_message(t('Unredeemed @n item(s).', array('@n' => $qty)));
  }
  _uc_product_redeem_redirect_after($order->order_id);
}

/** Submit: partial OFF, Redeem all */
function uc_product_redeem_inline_row_submit_redeem_all($form, &$form_state) {
  if (isset($form_state['values']['order_id'])){
    $order = uc_order_load($form_state['values']['order_id']);
    $opid  = (int) $form_state['values']['opid'];
  }else if (isset($form_state['complete form'])){
    // IF validation errors is set on the element use complete form because `values` will not contain what we want
    $order = uc_order_load($form_state['complete form']['order_id']['#value']);
    $opid = (int) $form_state['complete form']['opid']['#value'];
  }

  if (!$order || !$opid){
    return drupal_set_message(t('Order ID or Order product ID not valid', array()));
  }  

  $items = uc_product_redeem_order_items($order->order_id);
  foreach ($items as $it) {
    if ($it->order_product_id == $opid && $it->remaining > 0) {
      if (uc_product_redeem_apply($order, $opid, $it->remaining, 'inline')) {
        drupal_set_message(t('Redeemed @n item(s).', array('@n' => $it->remaining)));
      }
      break;
    }
  }
  _uc_product_redeem_redirect_after($order->order_id);
}

/** Submit: partial OFF, Unredeem all */
function uc_product_redeem_inline_row_submit_unredeem_all($form, &$form_state) {
  if (isset($form_state['values']['order_id'])){
    $order = uc_order_load($form_state['values']['order_id']);
    $opid  = (int) $form_state['values']['opid'];
  }else if (isset($form_state['complete form'])){
    // IF validation errors is set on the element use complete form because `values` will not contain what we want
    $order = uc_order_load($form_state['complete form']['order_id']['#value']);
    $opid = (int) $form_state['complete form']['opid']['#value'];
  }

  if (!$order || !$opid){
    return drupal_set_message(t('Order ID or Order product ID not valid', array()));
  }

  $items = uc_product_redeem_order_items($order->order_id);
  foreach ($items as $it) {
    if ($it->order_product_id == $opid && $it->redeemed > 0) {
      if (uc_product_redeem_apply($order, $opid, -$it->redeemed, 'inline')) {
        drupal_set_message(t('Unredeemed @n item(s).', array('@n' => $it->redeemed)));
      }
      break;
    }
  }
  _uc_product_redeem_redirect_after($order->order_id);
}

/**
 * "Redeem all remaining" button for the entire order.
 */
function uc_product_redeem_all_form($form, &$form_state, $order, $items) {
  $remaining_total = 0;
  foreach ($items as $it) $remaining_total += $it->remaining;

  $form['order_id'] = array('#type' => 'hidden', '#value' => $order->order_id);
  $form['remaining_total'] = array('#type' => 'hidden', '#value' => $remaining_total);

  if ($remaining_total > 0) {
    $form['actions'] = array('#type' => 'actions');
    $form['actions']['redeem_all'] = array(
      '#type' => 'submit',
      '#value' => t('Mark all items as redeemed (@n)', array('@n' => $remaining_total)),
      '#submit' => array('uc_product_redeem_all_form_submit'),
    );
  }
  return $form;
}

function uc_product_redeem_all_form_submit($form, &$form_state) {
  if (isset($form_state['values']['order_id'])){
    $order = uc_order_load($form_state['values']['order_id']);
  }else if (isset($form_state['complete form'])){
    // IF validation errors is set on the element use complete form because `values` will not contain what we want
    $order = uc_order_load($form_state['complete form']['order_id']['#value']);
  }

  // Basic validation
  if (!$order){
    return drupal_set_message(t('Order ID or Order product ID not valid', array()));
  }

  $items = uc_product_redeem_order_items($order->order_id);
  $did = 0;
  foreach ($items as $it) {
    if ($it->remaining > 0) {
      if (uc_product_redeem_apply($order, $it->order_product_id, $it->remaining, 'bulk')) {
        $did += $it->remaining;
      }
    }
  }
  if ($did > 0) {
    drupal_set_message(t('Redeemed @n item(s) in total.', array('@n' => $did)));
  } else {
    drupal_set_message(t('Nothing to redeem.'), 'warning');
  }
  _uc_product_redeem_redirect_after($order->order_id);
}

/**
 * Helper: post-action redirect.
 */
function _uc_product_redeem_redirect_after($order_id) {
  $mode = variable_get('uc_product_redeem_redirect', 'stay');
  switch ($mode) {
    case 'order':
      drupal_goto('admin/store/orders/' . $order_id);
      break;
    case 'stay':
      $qs = array();
      if (isset($_GET['k'])) $qs['k'] = $_GET['k'];
      drupal_goto('product-redeem/' . $order_id . '/redeem', array('query' => $qs));
      break;
    case 'none':
    default:
      // Allow standard redirect after submit to refresh.
      break;
  }
}

/**
 * Render a collapsible order comments panel.
 */
function uc_product_redeem_order_comments_markup($order_id, $open = FALSE) {
  $rows = array();
  $res = db_query("SELECT message, order_status, created FROM {uc_order_comments} WHERE order_id = :oid ORDER BY created DESC", array(':oid' => $order_id));
  foreach ($res as $row) {
    $rows[] = array(
      format_date($row->created, 'short'),
      //check_plain($row->order_status),
      check_plain($row->message),
    );
  }
  $table = theme('table', array(
    'header' => array(
      t('Date'), 
      // t('Status'), // status is not needed 
      t('Message')
    ),
    'rows' => $rows,
    'attributes' => array(
      'class' => array('ucpr-comments-table'),
      'style' => array(
        'text-align:left;',
      ),
    ),
  ));
  $attr_open = $open ? ' open' : '';
  return '<details class="ucpr__comments ucpr__order-details"' . $attr_open . '><summary>' . t('Order comments') . '</summary>' . $table . '</details>';
}

/**
 * Collapsible "Add note" form (writes one comment only).
 */
function uc_product_redeem_note_form($form, &$form_state, $order_id) {
  $form['order_id'] = array('#type' => 'value', '#value' => (int) $order_id);

  $form['ucpr_redeem_note'] = array(
    '#type' => 'container',
    '#attributes' => array(
      'class' => ['ucpr-note']
    ),
    '#prefix' => '<details class="ucpr__note ucpr__order-details"><summary>' . t('Add note') . '</summary>',
    '#suffix' => '</details>',
  );

  $form['ucpr_redeem_note']['note'] = array(
    '#type' => 'textarea',
    '#title' => t('Note'),
    '#title_display' => 'invisible',
    '#rows' => 3,
    '#attributes' => array('placeholder' => t('Optional note to attach to the order…')),
  );

  $form['ucpr_redeem_note']['actions'] = array('#type' => 'actions');
  $form['ucpr_redeem_note']['actions']['save'] = array(
    '#type' => 'submit',
    '#value' => t('Add note'),
    '#submit' => array('uc_product_redeem_note_form_submit'),
  );

  return $form;
}

function uc_product_redeem_note_form_submit($form, &$form_state) {
  $order_id = (int) $form_state['values']['order_id'];
  $note = trim($form_state['values']['note']);
  if ($note === '') {
    drupal_set_message(t('Note is empty.'), 'warning');
    return;
  }
  global $user;
  db_insert('uc_order_comments')
    ->fields(array(
      'order_id' => $order_id,
      'uid' => $user->uid,
      'order_status' => 'completed',
      'notified' => 0,
      'message' => $note,
      'created' => REQUEST_TIME,
    ))
    ->execute();
  drupal_set_message(t('Note added to order.'));
  _uc_product_redeem_redirect_after($order_id);
}