<?php

declare(strict_types=1);

namespace Drupal\ui_patterns_settings;

use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Trait CSSWidgetTrait.
 *
 * Provides methods to handle CSS widget settings and configurations.
 */
trait CSSWidgetTrait {

  use StringTranslationTrait;

  /**
   * Alters the options based on widget settings.
   *
   * @param array $options
   *   The original options array.
   * @param array $widget_settings
   *   The widget settings array.
   *
   * @return array
   *   The altered options array.
   */
  private function enumAlterOptions(array $options, array $widget_settings): array {
    $overwrite_options = $widget_settings['overwrite'] ?? [];
    foreach ($overwrite_options as $overwrite_option_id => $overwrite_option) {
      if (is_string($overwrite_option_id)) {
        $overwrite_option_id = str_replace('|||', '.', $overwrite_option_id);
      }
      if ($overwrite_option['enable'] == '0') {
        unset($options[$overwrite_option_id]);
        continue;
      }
      if (!empty($overwrite_option['title'])) {
        $options[$overwrite_option_id] = $overwrite_option['title'];
      }
    }
    return $options;
  }

  /**
   * Retrieves the available widget types.
   *
   * This method returns an array of widget types, each with its configuration
   * such as type, label, wrapper classes, and libraries.
   *
   * @return array
   *   An array of widget types.
   */
  private function getWidgetTypes(): array {
    return [
      'select' => [
        'type' => 'select',
        'label' => $this->t('Select'),
      ],
      'colors' => [
        'type' => 'radios',
        'wrapper_classes' => 'uips-layout-color',
        'library' => ['ui_patterns_settings/color'],
        'label' => $this->t('Colors'),
      ],
      'inline_grid' => [
        'type' => 'radios',
        'wrapper_classes' => 'uips-layout-grid uips-layout-grid-inline',
        'library' => ['ui_patterns_settings/grid'],
        'label' => $this->t('Inline grid/buttons'),
      ],
      '1_grid' => [
        'type' => 'radios',
        'wrapper_classes' => 'uips-layout-grid uips-layout-grid-1',
        'library' => ['ui_patterns_settings/grid'],
        'label' => $this->t('1 Column grid'),
      ],
      '3_grid' => [
        'type' => 'radios',
        'wrapper_classes' => 'uips-layout-grid uips-layout-grid-3',
        'label' => $this->t('3 Column grid'),
        'library' => ['ui_patterns_settings/grid'],
      ],
      '4_grid' => [
        'type' => 'radios',
        'wrapper_classes' => 'uips-layout-grid uips-layout-grid-4',
        'label' => $this->t('4 Column grid'),
        'library' => ['ui_patterns_settings/grid'],
      ],
    ];
  }

  /**
   * Builds the CSS settings form element.
   *
   * @param array $element
   *   The form element array.
   * @param array $widget_settings
   *   The widget settings array.
   *
   * @return array
   *   The modified form element array.
   */
  public function cssSettingsForm(array $element, array $widget_settings): array {
    $element['#options'] = $this->enumAlterOptions($element['#options'], $widget_settings);
    $this->addCssRequired($element, $widget_settings);
    $widget = $widget_settings['widget'];
    $widget_configs = $this->getWidgetTypes();
    $widget_config = $widget_configs[$widget];
    $element['#type'] = $widget_config['type'];
    if (empty($element['#default_value']) && !empty($widget_settings['default_option'])) {
      $element['#default_value'] = $widget_settings['default_option'];
    }

    $configure = $this->getEnumConfiguration($widget_settings);
    $configure['css']['wrapper_classes'] .= ' ' . ($widget_config['wrapper_classes'] ?? '');
    $element['#enum_widget'] = $configure;
    if ($element['#type'] === 'radios') {
      $element['#theme_wrappers'] = ['radios', 'fieldset'];
      $element['#pre_render'][] = function ($element) use ($configure) {
        foreach (Element::children($element) as $key) {
          $element[$key]['#theme_wrappers'] = ['ui_patterns_settings_radio_wrapper_widget'];
          $element[$key]['#theme'] = ['ui_patterns_settings_radio_widget'];
          $element[$key]['#enum_widget'] = $configure;
        }
        return $element;
      };
    }
    $element['#attached']['library'] = $widget_config['library'] ?? [];
    return $element;
  }

  /**
   * Add required property and title.
   *
   * @param array $form_element
   *   The form element.
   * @param array $widget_settings
   *   The widget settings.
   *
   * @see \Drupal\ui_patterns\Element\ComponentPropForm::addRequired()
   */
  protected function addCssRequired(&$form_element, array $widget_settings): void {
    $form_element['#title'] = $widget_settings['title'];
    $form_element['#title_display'] = $widget_settings['title_display'];
    $form_element['#description'] = $widget_settings['description'];
    $form_element['#description_display'] = $widget_settings['description_display'];
    $form_element['#required'] = $widget_settings['required'];
  }

  /**
   * Retrieves the enum configuration from widget settings.
   *
   * @param array $widget_settings
   *   The widget settings array.
   *
   * @return array
   *   The enum configuration array.
   */
  private function getEnumConfiguration(array $widget_settings): array {
    return [
      'widget' => $widget_settings['widget'],
      'overwrite' => $widget_settings['overwrite'],
      'css' => $widget_settings['css'],
      'default_option' => $widget_settings['default_option'],
    ];
  }

  /**
   * Base widget settings form.
   */
  public function baseWidgetSettingsForm(array $widget_settings = []): array {
    return [
      'required' =>
        [
          '#title' => $this->t('Required'),
          '#type' => 'checkbox',
          '#default_value' => $widget_settings['required'] ?? FALSE,
        ],
      'title' =>
        [
          '#title' => $this->t('Title'),
          '#type' => 'textfield',
          '#default_value' => $widget_settings['title'] ?? '',
        ],
      'title_display' =>
        [
          '#title' => $this->t('Title display'),
          '#type' => 'select',
          '#options' => [
            'before' => $this->t('Before'),
            'after' => $this->t('After'),
            'invisible' => $this->t('Invisible'),
          ],
          '#default_value' => $widget_settings['title_display'] ?? 'before',
        ],
      'description' =>
        [
          '#title' => $this->t('Description'),
          '#type' => 'textfield',
          '#default_value' => $widget_settings['description'] ?? '',
        ],
      'description_display' =>
        [
          '#title' => $widget_settings['Description display'],
          '#type' => 'select',
          '#options' => [
            'before' => $this->t('Before'),
            'after' => $this->t('After'),
            'invisible' => $this->t('Invisible'),
          ],
          '#default_value' => $widget_settings['description_display'] ?? 'after',
        ],
    ];
  }

  /**
   * Builds the CSS widget settings form.
   *
   * @param array $element
   *   The form element array.
   * @param array $widget_settings
   *   The widget settings array.
   * @param array $options
   *   The options array for the widget.
   *
   * @return array
   *   The modified form element array.
   */
  public function cssWidgetSettingsForm(array $element, array $widget_settings, array $options): array {
    $configure = $this->getEnumConfiguration($widget_settings);
    $widget_options = array_map(fn($widget_type) => $widget_type['label'], $this->getWidgetTypes());
    $element['widget'] = [
      '#type' => 'select',
      '#title' => $this->t('Widget'),
      '#description' => $this->t('Enum widget type.'),
      '#options' => $widget_options,
      '#default_value' => $configure['widget'] ?? 'select',
    ];
    $element += $this->baseWidgetSettingsForm($widget_settings);
    $element['css']['wrapper_classes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Wrapper CSS classes'),
      '#default_value' => $configure['css']['wrapper_classes'] ?? '',
      '#description' => $this->t('Classes assigned to wrapping select element'),
    ];
    $element['default_option'] = [
      '#type' => 'select',
      '#title' => $this->t('Default option'),
      '#default_value' => $configure['default_option'] ?? '',
      '#description' => $this->t('Preselected option'),
      '#options' => $options,
      '#empty_option' => $this->t('None'),
    ];

    foreach ($options as $option_id => $label) {
      if (is_string($option_id)) {
        $option_id = str_replace('.', '|||', $option_id);
      }
      $element['overwrite'][$option_id] = [
        '#type' => 'details',
        '#title' => $this->t('Option: %label', ['%label' => $label]),
      ];
      $element['overwrite'][$option_id]['enable'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable'),
        '#default_value' => $configure['overwrite'][$option_id]['enable'] ?? TRUE,
      ];
      $element['overwrite'][$option_id]['title'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Title'),
        '#default_value' => $configure['overwrite'][$option_id]['title'] ?? '',
      ];
      $element['overwrite'][$option_id]['hide_title'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Hide title'),
        '#default_value' => $configure['overwrite'][$option_id]['hide_title'] ?? '0',
      ];

      $element['overwrite'][$option_id]['icon'] = [
        '#type' => 'icon_autocomplete',
        '#title' => $this->t('Icon'),
        '#return_id' => TRUE,
        '#description' => $this->t('May not work with your widget.'),
        '#default_value' => $configure['overwrite'][$option_id]['icon']['target_id'] ?? $configure['overwrite'][$option_id]['icon']['icon_id'] ?? '',
      ];

      $element['overwrite'][$option_id]['component_wrapper'] = [
        '#type' => 'details',
        '#title' => $this->t('Component'),
        'component' => [
          '#type' => 'component_form',
          '#title' => $this->t('Component'),
          '#component_id' => $element['#component_id'],
          '#component_validation' => FALSE,
          '#component_required' => FALSE,
          '#default_value' => $configure['overwrite'][$option_id]['component_wrapper']['component'] ?? '',
        ],
      ];

      $element['overwrite'][$option_id]['css'] = [
        '#type' => 'textfield',
        '#title' => $this->t('CSS'),
        '#default_value' => $configure['overwrite'][$option_id]['css'] ?? '',
      ];
    }
    return $element;
  }

}
