<?php

declare(strict_types=1);

namespace Drupal\ui_patterns_settings\Plugin\UiPatterns\Source;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Template\Attribute;
use Drupal\ui_patterns\Attribute\Source;
use Drupal\ui_patterns\PluginWidgetSettingsInterface;
use Drupal\ui_patterns\SourcePluginBase;
use Drupal\ui_patterns\WidgetSettingTrait;
use Drupal\ui_patterns_settings\CSSWidgetTrait;
use Drupal\ui_styles\StylePluginManagerInterface;

/**
 * UI Styles CSS Widget.
 *
 * This class provides a widget for displaying configured UI styles
 * as grids, colors, etc., and allows for customization of CSS settings.
 */
#[Source(
  id: 'ui_styles_css_widget',
  label: new TranslatableMarkup('UI Styles CSS widget'),
  description: new TranslatableMarkup('Display configured UI Styles as grid / colors / etc.'),
  prop_types: ['attributes'],
  tags: ['widget']
)]
class UiStylesWidget extends SourcePluginBase implements PluginWidgetSettingsInterface {

  use WidgetSettingTrait;
  use CSSWidgetTrait;

  /**
   * {@inheritdoc}
   */
  public function getPropValue(): mixed {
    $attributes = new Attribute();
    $plugins = array_keys($this->settings);
    foreach ($plugins as $plugins_id) {
      /** @var \Drupal\ui_styles\Definition\StyleDefinition $style_definition */
      $style_definition = self::stylesManager()->getDefinition($plugins_id);
      $options = $style_definition->getOptions();
      $option = $options[$this->settings[$plugins_id]]['class'] ?? $this->settings[$plugins_id];
      $attributes->addClass($option ?? '');
    }

    return $attributes;
  }

  /**
   * Gets the style plugin manager service.
   */
  protected static function stylesManager(): StylePluginManagerInterface {
    return \Drupal::service('plugin.manager.ui_styles');
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $style_plugin_manager = self::stylesManager();
    $form = parent::settingsForm($form, $form_state);
    $definitions = $style_plugin_manager->getDefinitions();
    foreach ($definitions as $id => $definition) {
      $form[$id] = [
        '#title' => $definition->getLabel(),
        '#options' => $definition->getOptionsAsOptions(),
        '#empty_option' => $definition->getEmptyOption(),
        '#default_value' => $this->getSetting($id),
      ];
      $enable = $this->widgetSettings[$id]['enable'] ?? TRUE;
      if ($enable != '0') {
        $form[$id] = $this->cssSettingsForm($form[$id], $this->widgetSettings[$id] ?? []);

      }
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function widgetSettingsForm(array $form, FormStateInterface $form_state): array {
    $style_plugin_manager = self::stylesManager();
    $form = [];
    $definitions = $style_plugin_manager->getDefinitions();

    foreach ($definitions as $id => $definition) {
      $options = $definition->getOptionsAsOptions();
      $form[$id] = [
        '#type' => 'details',
        '#title' => $definition->getLabel(),
        '#collapsed' => TRUE,
      ];

      $widget_settings = $this->widgetSettings[$id] ?? [];
      if (!isset($widget_settings['title'])) {
        $widget_settings['title'] = $definition->getLabel();
      }
      if (!isset($widget_settings['description'])) {
        $widget_settings['description'] = $definition->getDescription();
      }
      $form[$id] = $this->cssWidgetSettingsForm($form[$id], $widget_settings, $options);
      $form[$id]['enable'] = [
        '#weight' => -10,
        '#type' => 'checkbox',
        '#title' => $this->t('Enable'),
        '#default_value' => $this->widgetSettings[$id]['enable'] ?? TRUE,
      ];
      $form[$id]['required'] = [
        '#weight' => -10,
        '#type' => 'checkbox',
        '#title' => $this->t('Required'),
        '#default_value' => $this->widgetSettings[$id]['required'] ?? FALSE,
      ];
    }
    return $form;
  }

}
