<?php

declare(strict_types=1);

namespace Drupal\ui_patterns_settings\Plugin\UiPatterns\Source;

use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\Entity\Media;
use Drupal\ui_patterns\Attribute\Source;
use Drupal\ui_patterns\PluginWidgetSettingsInterface;
use Drupal\ui_patterns\SourcePluginBase;
use Drupal\ui_patterns\WidgetSettingTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the source.
 */
#[Source(
  id: 'media',
  label: new TranslatableMarkup('Media'),
  description: new TranslatableMarkup('Provides a Media source.'),
  prop_types: ['slot']
)]
class MediaSource extends SourcePluginBase implements PluginWidgetSettingsInterface {

  use WidgetSettingTrait {
    widgetSettingsForm as private parentWidgetSettingsForm;
  }

  /**
   * The entity type manager.
   */
  protected ?EntityTypeManager $entityTypeManager;

  /**
   * The entity display repository.
   */
  protected ?EntityDisplayRepositoryInterface $entityDisplayRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $plugin = parent::create(
      $container,
      $configuration,
      $plugin_id,
      $plugin_definition
    );
    $plugin->entityTypeManager = $container->get('entity_type.manager');
    $plugin->entityDisplayRepository = $container->get('entity_display.repository');
    return $plugin;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'media' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getPropValue(): mixed {

    $media = $this->getMedia();
    if ($media === NULL) {
      return '';
    }
    $view_mode = $this->getWidgetSetting('view_mode') ?? 'default';
    $view_mode_builder = $this->entityTypeManager->getViewBuilder('media');
    return $view_mode_builder->view($media, $view_mode);

  }

  /**
   * Returns the referenced media object.
   */
  protected function getMedia(): ?Media {
    $media_configuration = $this->getSetting('media');
    $media_id = $media_configuration['media_library_selection'] ?? $media_configuration;
    return $media_id !== NULL && is_numeric($media_id) ? Media::load($media_id) : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function widgetSettingsForm(array $form, FormStateInterface $form_state): array {
    $form = $this->parentWidgetSettingsForm($form, $form_state);
    // @phpstan-ignore-line
    /** @var \Drupal\media\Entity\MediaType[] $bundles */
    $bundles = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
    $bundle_options = [];
    foreach ($bundles as $bundle_id => $bundle) {
      $bundle_options[$bundle_id] = $bundle->label();
    }
    $form['allowed_bundles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Bundles'),
      '#options' => $bundle_options,
      '#default_value' => $this->getWidgetSetting('allowed_bundles') ?? FALSE,
    ];
    $form['view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('View Modes'),
      '#options' => $this->entityDisplayRepository->getViewModeOptions('media'),
      '#default_value' => $this->getWidgetSetting('view_mode') ?? FALSE,
    ];
    $form['show_edit_button'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Edit button'),
      '#default_value' => $this->getWidgetSetting('show_edit_button') ?? FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $form = parent::settingsForm($form, $form_state);
    $configure = $this->getConfiguration()['configure'];
    $media = $this->getMedia();
    $media_id = $media?->id();

    $form['media'] = [
      '#type' => 'media_library',
      '#default_value' => $media_id,
    ];

    $this->addRequired($form['media']);
    if ($media !== NULL && $this->getWidgetSetting('show_edit_button') === TRUE) {
      $form['media']['#ui_patterns_media_edit_button'] = TRUE;
    }

    $allowed_bundles = $this->getWidgetSetting('allowed_bundles') ?? ['image' => 'image'];
    $allowed_bundles = array_keys(array_filter($allowed_bundles));
    if (!empty($allowed_bundles)) {
      $form['media']['#allowed_bundles'] = $allowed_bundles;
    }
    else {
      $form['media']['#allowed_bundles'] = ['image'];
    }
    $form['media']['#cardinality'] = $configure['cardinality'] ?? 1;
    return $form;
  }

}
