<?php

declare(strict_types=1);

namespace Drupal\ui_suite_bootstrap\Hook;

use Drupal\Core\Cache\CacheCollectorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Add theme settings.
 */
class ThemeSettings {

  use StringTranslationTrait;

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    #[Autowire(service: 'library.discovery')]
    protected CacheCollectorInterface $libraryDiscovery,
  ) {}

  /**
   * Implements hook_form_system_theme_settings_alter().
   */
  #[Hook('form_system_theme_settings_alter')]
  public function alter(array &$form, FormStateInterface $formState): void {
    if (isset($form['config_key']['#value']) && \is_string($form['config_key']['#value'])) {
      $config_key = $form['config_key']['#value'];
    }
    else {
      return;
    }

    $form['ui_suite_bootstrap'] = [
      '#type' => 'details',
      '#title' => $this->t('Bootstrap'),
      '#open' => TRUE,
    ];

    $form['ui_suite_bootstrap']['container'] = [
      '#type' => 'select',
      '#title' => $this->t('Page container'),
      '#description' => $this->t('Select an option for <a href=":url">Bootstrap containers</a>.', [
        ':url' => 'https://getbootstrap.com/docs/5.3/layout/containers/',
      ]),
      '#options' => [
        'container' => $this->t('Container'),
        'container-sm' => $this->t('Container small'),
        'container-md' => $this->t('Container medium'),
        'container-lg' => $this->t('Container large'),
        'container-xl' => $this->t('Container x-large'),
        'container-xxl' => $this->t('Container xx-large'),
        'container-fluid' => $this->t('Container fluid'),
      ],
      '#config_target' => $config_key . ':container',
    ];

    $form['ui_suite_bootstrap']['library'] = [
      '#type' => 'details',
      '#title' => $this->t('Library'),
      '#tree' => TRUE,
    ];
    $form['ui_suite_bootstrap']['library']['js_loading'] = [
      '#type' => 'select',
      '#title' => $this->t('JavaScript'),
      '#description' => $this->t('If left empty, it is assumed that you have custom code or sub-theme altering or extending ui_suite_bootstrap/framework library to load JS your own way.'),
      '#options' => [
        'ui_suite_bootstrap/framework_js' => $this->t('Local'),
        'ui_suite_bootstrap/framework_js_cdn' => $this->t('CDN'),
      ],
      '#empty_value' => '',
      '#config_target' => $config_key . ':library.js_loading',
      '#after_build' => [
        [static::class, 'filterLibraries'],
      ],
    ];
    $form['ui_suite_bootstrap']['library']['css_loading'] = [
      '#type' => 'select',
      '#title' => $this->t('Stylesheet'),
      '#description' => $this->t('If left empty, it is assumed that you have custom code or sub-theme altering or extending ui_suite_bootstrap/framework library to load CSS your own way.')
      . '<br>'
      . $this->t('Visit <a href=":url">Bootswatch\'s website</a> to get a preview of the themes.', [
        ':url' => 'https://bootswatch.com',
      ]),
      '#options' => [
        'ui_suite_bootstrap/framework_css_bootstrap' => $this->t('Bootstrap'),
        'ui_suite_bootstrap/framework_css_cdn_bootstrap' => $this->t('Bootstrap (CDN)'),
        $this->t('Bootswatch (local)')->render() => [
          'ui_suite_bootstrap/framework_css_bootswatch_brite' => $this->t('Brite'),
          'ui_suite_bootstrap/framework_css_bootswatch_cerulean' => $this->t('Cerulean'),
          'ui_suite_bootstrap/framework_css_bootswatch_cosmo' => $this->t('Cosmo'),
          'ui_suite_bootstrap/framework_css_bootswatch_cyborg' => $this->t('Cyborg'),
          'ui_suite_bootstrap/framework_css_bootswatch_darkly' => $this->t('Darkly'),
          'ui_suite_bootstrap/framework_css_bootswatch_flatly' => $this->t('Flatly'),
          'ui_suite_bootstrap/framework_css_bootswatch_journal' => $this->t('Journal'),
          'ui_suite_bootstrap/framework_css_bootswatch_litera' => $this->t('Litera'),
          'ui_suite_bootstrap/framework_css_bootswatch_lumen' => $this->t('Lumen'),
          'ui_suite_bootstrap/framework_css_bootswatch_lux' => $this->t('Lux'),
          'ui_suite_bootstrap/framework_css_bootswatch_materia' => $this->t('Materia'),
          'ui_suite_bootstrap/framework_css_bootswatch_minty' => $this->t('Minty'),
          'ui_suite_bootstrap/framework_css_bootswatch_morph' => $this->t('Morph'),
          'ui_suite_bootstrap/framework_css_bootswatch_pulse' => $this->t('Pulse'),
          'ui_suite_bootstrap/framework_css_bootswatch_quartz' => $this->t('Quartz'),
          'ui_suite_bootstrap/framework_css_bootswatch_sandstone' => $this->t('Sandstone'),
          'ui_suite_bootstrap/framework_css_bootswatch_simplex' => $this->t('Simplex'),
          'ui_suite_bootstrap/framework_css_bootswatch_sketchy' => $this->t('Sketchy'),
          'ui_suite_bootstrap/framework_css_bootswatch_slate' => $this->t('Slate'),
          'ui_suite_bootstrap/framework_css_bootswatch_solar' => $this->t('Solar'),
          'ui_suite_bootstrap/framework_css_bootswatch_spacelab' => $this->t('Spacelab'),
          'ui_suite_bootstrap/framework_css_bootswatch_superhero' => $this->t('Superhero'),
          'ui_suite_bootstrap/framework_css_bootswatch_united' => $this->t('United'),
          'ui_suite_bootstrap/framework_css_bootswatch_vapor' => $this->t('Vapor'),
          'ui_suite_bootstrap/framework_css_bootswatch_yeti' => $this->t('Yeti'),
          'ui_suite_bootstrap/framework_css_bootswatch_zephyr' => $this->t('Zephyr'),
        ],
        $this->t('Bootswatch (CDN)')->render() => [
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_brite' => $this->t('Brite'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_cerulean' => $this->t('Cerulean'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_cosmo' => $this->t('Cosmo'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_cyborg' => $this->t('Cyborg'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_darkly' => $this->t('Darkly'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_flatly' => $this->t('Flatly'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_journal' => $this->t('Journal'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_litera' => $this->t('Litera'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_lumen' => $this->t('Lumen'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_lux' => $this->t('Lux'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_materia' => $this->t('Materia'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_minty' => $this->t('Minty'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_morph' => $this->t('Morph'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_pulse' => $this->t('Pulse'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_quartz' => $this->t('Quartz'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_sandstone' => $this->t('Sandstone'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_simplex' => $this->t('Simplex'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_sketchy' => $this->t('Sketchy'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_slate' => $this->t('Slate'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_solar' => $this->t('Solar'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_spacelab' => $this->t('Spacelab'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_superhero' => $this->t('Superhero'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_united' => $this->t('United'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_vapor' => $this->t('Vapor'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_yeti' => $this->t('Yeti'),
          'ui_suite_bootstrap/framework_css_cdn_bootswatch_zephyr' => $this->t('Zephyr'),
        ],
      ],
      '#empty_value' => '',
      '#config_target' => $config_key . ':library.css_loading',
      '#after_build' => [
        [static::class, 'filterLibraries'],
      ],
    ];

    $form['#submit'][] = static::class . ':submitForm';
  }

  /**
   * Submit callback.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    /** @var string $config_key */
    $config_key = $form_state->getValue('config_key');
    $theme_config = $this->configFactory->get($config_key);

    $checked_keys = [
      'library.js_loading' => ['library', 'js_loading'],
      'library.css_loading' => ['library', 'css_loading'],
    ];

    foreach ($checked_keys as $config_path => $form_path) {
      if ($theme_config->get($config_path) != $form_state->getValue($form_path)) {
        $this->libraryDiscovery->clear();
        return;
      }
    }
  }

  /**
   * Remove invalid libraries from select options.
   *
   * @param array{"#options": array} $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The modified element.
   */
  public static function filterLibraries(array $element, FormStateInterface $form_state): array {
    $element['#options'] = static::clearLibraries($element['#options']);
    return $element;
  }

  /**
   * Filter invalid libraries.
   *
   * @param array $options
   *   The options to filter.
   *
   * @return array
   *   The changed options.
   */
  protected static function clearLibraries(array $options): array {
    /** @var \Drupal\Core\Asset\LibraryDiscoveryInterface $library_discovery */
    $library_discovery = \Drupal::service('library.discovery');
    foreach ($options as $option => $value) {
      if (\is_array($value)) {
        $options[$option] = static::clearLibraries($value);
        if (empty($options[$option])) {
          unset($options[$option]);
        }
        continue;
      }

      $library_name = \explode('/', $option);
      if (!isset($library_name[1])) {
        continue;
      }
      $library = $library_discovery->getLibraryByName($library_name[0], $library_name[1]);
      // @phpstan-ignore-next-line
      if ($library && !static::isLibraryValid($library)) {
        unset($options[$option]);
      }
    }

    return $options;
  }

  /**
   * Check if library files are accessible.
   *
   * @param array{js?: array{array{type: string, data: string}}, css?: array{array{type: string, data: string}}, dependencies?: string[]} $library
   *   A library definition.
   *
   * @return bool
   *   TRUE if the library files are accessible.
   *
   * @see https://www.drupal.org/project/drupal/issues/2231385
   */
  protected static function isLibraryValid(array $library): bool {
    if (\array_key_exists('js', $library)) {
      foreach ($library['js'] as $js) {
        if ($js['type'] == 'file') {
          if (!\file_exists(DRUPAL_ROOT . '/' . $js['data'])) {
            return FALSE;
          }
        }
      }
    }

    if (\array_key_exists('css', $library)) {
      foreach ($library['css'] as $css) {
        if ($css['type'] == 'file') {
          if (!\file_exists(DRUPAL_ROOT . '/' . $css['data'])) {
            return FALSE;
          }
        }
      }
    }

    if (\array_key_exists('dependencies', $library)) {
      foreach ($library['dependencies'] as $dependency) {
        $parts = \explode('/', $dependency, (int) 2);
        $dependencyLibrary = \Drupal::service('library.discovery')->getLibraryByName($parts[0], $parts[1]);
        // @phpstan-ignore-next-line
        if ($dependencyLibrary && !static::isLibraryValid($dependencyLibrary)) {
          return FALSE;
        }
      }
    }

    return TRUE;
  }

}
