<?php

declare(strict_types=1);

namespace Drupal\Tests\ui_suite_dsfr\Kernel\Source;

use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\system\Entity\Menu;
use Drupal\Tests\ui_suite_dsfr\Kernel\KernetTestBase;

/**
 * Test AttributesWidget.
 *
 * @coversDefaultClass \Drupal\ui_patterns\Plugin\UiPatterns\Source\AttributesWidget
 * @group ui_patterns
 */
class NavMenuTest extends KernetTestBase {


  /**
   * The menu link plugin manager.
   *
   * @var \Drupal\Core\Menu\MenuLinkManagerInterface
   */
  protected $menuLinkManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    // @todo Change the autogenerated stub
    parent::setUp();
    $this->enableModules(['menu_link_content', 'router_test', 'entity_test']);
    $this->installEntitySchema('menu_link_content');
    $this->installEntitySchema('user');
    $this->menuLinkManager = \Drupal::service('plugin.manager.menu.link');
    Menu::create([
      'id' => 'menu-test',
      'label' => 'Test menu',
      'description' => 'Description text',
    ])->save();
  }

  /**
   * Create a simple hierarchy of links.
   */
  public function createLinkHierarchy(string $module = 'menu_test') : array {
    // First remove all the menu links in the menu.
    $this->menuLinkManager->deleteLinksInMenu('menu-test');

    // Then create a simple link hierarchy:
    // - parent
    //   - child-1
    //     - child-1-1
    //     - child-1-2
    //   - child-2.
    $base_options = [
      'provider' => $module,
      'menu_name' => 'menu-test',
    ];

    $parent = $base_options + [
      'title' => 'parent',
      'link' => ['uri' => 'internal:/menu-test/hierarchy/parent'],
    ];
    $link = MenuLinkContent::create($parent);
    $link->save();
    $links['parent'] = $link->getPluginId();

    $child_1 = $base_options + [
      'title' => 'child-1',
      'link' => ['uri' => 'internal:/menu-test/hierarchy/parent/child-1'],
      'parent' => $links['parent'],
      'expanded' => TRUE,
    ];
    $link = MenuLinkContent::create($child_1);
    $link->save();
    $links['child-1'] = $link->getPluginId();

    $child_1_1 = $base_options + [
      'title' => 'child-1-1',
      'link' => ['uri' => 'internal:/menu-test/hierarchy/parent/child-1/child-1-1'],
      'parent' => $links['child-1'],
    ];
    $link2 = MenuLinkContent::create($child_1_1);
    $link2->save();
    $links['child-1-1'] = $link2->getPluginId();
    $link->get('expanded')->value = TRUE;
    $link->save();

    $child_1_2 = $base_options + [
      'title' => 'child-1-2',
      'link' => ['uri' => 'route:<nolink>'],
      'parent' => $links['child-1'],
    ];
    $link = MenuLinkContent::create($child_1_2);
    $link->save();
    $links['child-1-2'] = $link->getPluginId();

    $child_2 = $base_options + [
      'title' => 'child-2',
      'link' => ['uri' => 'route:<nolink>'],
      'parent' => $links['parent'],
    ];
    $link = MenuLinkContent::create($child_2);
    $link->save();
    $links['child-2'] = $link->getPluginId();
    return $links;
  }

  /**
   * Test Menu.
   */
  public function testMenuTreeHieracrhy(): void {
    $this->menuLinkManager->deleteLinksInMenu('menu-test');
    $links = $this->createLinkHierarchy('ui_suite_dsfr');
    $render_array = [
      "#type" => "component",
      '#component' => 'ui_suite_dsfr:nav_menu',
      '#ui_patterns' => [
        'component_id' => 'ui_suite_dsfr:nav_menu',
        "props" => [
          "items" => [
            "source_id" => "menu",
            "source" => [
              "menu" => "menu-test",
              "level" => '1',
              "depth" => '0',
            ],
          ],
        ],
      ],
    ];
    $rendered_value = \Drupal::service('renderer')->renderRoot($render_array);
    $rendered_value = self::normalizeMarkupString((string) $rendered_value);
    // Level 2 children are not printed.
    $this->assertStringNotContainsString('child-1-1', $rendered_value);
    $this->assertStringNotContainsString('child-1-2', $rendered_value);
    // Route <nolink> is not printed.
    $this->assertStringNotContainsString('child-2', $rendered_value);
    // Should be printed.
    $this->assertStringContainsString('child-1', $rendered_value);
    $this->assertStringContainsString('parent', $rendered_value);
  }

}
