<?php

namespace Drupal\ultimate_table_field\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the ultimate table field widget.
 *
 * @FieldFormatter(
 *   id = "ultimate_table",
 *   label = @Translation("Ultimate table"),
 *   description = @Translation("Ultimate table default formatter."),
 *   field_types = {
 *     "ultimate_table"
 *   }
 * )
 */
class UltimateTableFieldFormatter extends FormatterBase {

  /**
   * Ultimate table cell field manager plugin.
   *
   * @var \Drupal\ultimate_table_field\UltimateTableCellFieldManager
   */
  protected $ultimateTableCellFieldManager;

  /**
   * Renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->ultimateTableCellFieldManager = $container->get('plugin.manager.ultimate_table_cell_field');
    $instance->renderer = $container->get('renderer');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'legend_position' => 'before',
      'table_theme' => 'default',
      'table_color' => '',
      'table_style' => [],
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $form['legend_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Legend position'),
      '#options' => [
        'before' => $this->t('Before table'),
        'after' => $this->t('After table'),
      ],
      '#default_value' => $this->getSetting('legend_position'),
    ];

    $form['table_theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Table color theme'),
      '#options' => [
        '' => $this->t('Default'),
        'black' => $this->t('Black'),
        'blue' => $this->t('Blue'),
        'brown' => $this->t('Brown'),
        'cyan' => $this->t('Cyan'),
        'gold' => $this->t('Golden'),
        'gray' => $this->t('Gray'),
        'green' => $this->t('Green'),
        'olive' => $this->t('Olive'),
        'orange' => $this->t('Orange'),
        'pink' => $this->t('Pink'),
        'purple' => $this->t('Purple'),
        'red' => $this->t('Red'),
        'silver' => $this->t('Silver'),
        'yellow' => $this->t('Yellow'),
      ],
      '#default_value' => $this->getSetting('table_theme'),
    ];

    $form['table_style'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Table styles'),
      '#options' => [
        'striped-odd' => $this->t('Striped rows (odd)'),
        'striped-even' => $this->t('Striped rows (even)'),
        'filled-head' => $this->t('Filled header'),
        'row-hover' => $this->t('Row hover effect'),
        'side-filled' => $this->t('Side filled'),
      ],
      '#default_value' => $this->getSetting('table_style'),
    ];

    // Make striped-odd and striped-even mutually exclusive.
    $form['table_style']['striped-odd']['#states'] = [
      'disabled' => [
        ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][table_style][striped-even]"]' => ['checked' => TRUE],
      ],
    ];
    $form['table_style']['striped-even']['#states'] = [
      'disabled' => [
        ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][table_style][striped-odd]"]' => ['checked' => TRUE],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];
    foreach ($items as $item) {
      $values = $item->value;
      $columns = $values['columns'] ?? [];
      $rows = $values['rows'] ?? [];
      $legend = $values['legend'] ?? '';

      $element = [];

      // Add legend based on position setting.
      if ($legend && $this->getSetting('legend_position') === 'before') {
        $element['legend'] = [
          '#type' => 'markup',
          '#markup' => $legend,
        ];
      }

      // Format columns.
      foreach ($columns as &$column_items) {
        foreach ($column_items as &$cell_item) {
          $cell_field_instance = $this->ultimateTableCellFieldManager->createInstance($cell_item['type']);
          $cell_item = $cell_field_instance->cellFieldFormatter($cell_item);
        }
      }

      foreach ($columns as &$column) {
        $column = $this->renderer->renderInIsolation($column);
      }

      // Add table.
      foreach ($rows as &$row) {
        foreach ($row as $j => $items) {
          foreach ($items as &$cell_item) {
            $cell_field_instance = $this->ultimateTableCellFieldManager->createInstance($cell_item['type']);
            $cell_item = $cell_field_instance->cellFieldFormatter($cell_item);
          }
          $row[$j] = $this->renderer->renderInIsolation($items);
        }
      }

      // Add table element.
      $element['table'] = [
        '#type' => 'table',
        '#header' => $columns,
        '#rows' => $rows,
        '#empty' => t('No entries available'),
        '#attributes' => [
          'class' => $this->getTableClasses(),
        ],
        '#prefix' => '<div class="table-ultimate">',
        '#suffix' => '</div>',
        '#attached' => [
          'library' => ['ultimate_table_field/table-themes'],
        ],
      ];

      // Add legend after table if configured.
      if ($legend && $this->getSetting('legend_position') === 'after') {
        $element['legend'] = [
          '#type' => 'markup',
          '#markup' => $legend,
        ];
      }

      $elements[] = $element;
    }
    return $elements;
  }

  /**
   * Gets the CSS classes for the table.
   *
   * @return array
   *   An array of CSS classes.
   */
  protected function getTableClasses() {
    $classes = ['custom-bordered-table'];

    // Add color theme.
    if ($theme = $this->getSetting('table_theme')) {
      $classes[] = 'table-' . $theme;
    }

    // Add selected styles.
    $styles = array_filter($this->getSetting('table_style'));
    $classes = array_merge($classes, array_keys($styles));

    return $classes;
  }

  /**
   * {@inheritDoc}
   */
  public function settingsSummary() {
    $styles = array_filter($this->getSetting('table_style'));
    $theme = $this->getSetting('table_theme');
    $theme = empty($theme) ? $this->t('Default') : $theme;
    return [
      'Applied color: ' . $theme,
      'Applied style: ' . implode(', ', $styles),
    ];
  }

}
