<?php

namespace Drupal\ultimate_table_field\Plugin\UltimateTable\CellField;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Entity\Element\EntityAutocomplete;
use Drupal\Core\Url;
use Drupal\ultimate_table_field\UltimateTableCellFieldBase;

/**
 * Provides link field cell form element.
 *
 * @UltimateTableCellField(
 *   id="link",
 *   label=@Translation("Default link"),
 * )
 */
class Link extends UltimateTableCellFieldBase {

  /**
   * {@inheritDoc}
   */
  public function buildCellField($default_value = NULL): array {
    $default_link = $default_value['uri'] ?? NULL;
    $default_link = $default_link && !UrlHelper::isExternal($default_link) ? static::getUriAsDisplayableString($default_link) : $default_link;
    return [
      '#type' => 'fieldset',
      '#tree' => TRUE,
      'uri' => [
        '#type' => 'entity_autocomplete',
        '#target_type' => 'node',
        '#title' => $this->t('Link URI'),
        '#description' => $this->t('Start typing the title of a piece of content to select it. You can also enter an internal path such as %add-node or an external URL such as %url. Enter %front to link to the front page. Enter %nolink to display link text only. Enter %button to display keyboard-accessible link text only.', [
          '%front' => '<front>',
          '%add-node' => '/node/add',
          '%url' => 'https://example.com',
          '%nolink' => '<nolink>',
          '%button' => '<button>',
        ]),
        '#element_validate' => [
          [
            'Drupal\link\Plugin\Field\FieldWidget\LinkWidget',
            'validateUriElement',
          ],
        ],
        '#attributes' => [
          // cspell:disable
          'data-autocomplete-first-character-blacklist' => '/#?',
          // cspell:enable
        ],
        '#process_default_value' => FALSE,
        '#default_value' => $default_link,
        '#required' => TRUE,
      ],
      'title' => [
        '#type' => 'textfield',
        '#title' => $this->t('Link text'),
        '#default_value' => $default_value['title'] ?? '',
      ],
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function cellFieldFormatter(array $item): array {
    try {
      $url = Url::fromUri($item[$this->id()]['uri'] ?? '');
    }
    catch (\InvalidArgumentException $e) {
      $url = Url::fromRoute('<none>');
    }
    $link_title = $item[$this->id()]['title'] ?? NULL;
    $link_title = $link_title ?? $url->toString();
    return [
      '#type' => 'container',
      'link' => [
        '#type' => 'link',
        '#title' => $link_title,
        '#url' => $url,
      ],
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function generateSummary($data = NULL): array {
    return [
      '#type' => 'markup',
      '#markup' => '<p><strong>Type:</strong> ' . $this->label() . ' <strong>Uri:</strong>' . substr(static::getUriAsDisplayableString($data[$this->id()]['uri']), 0, 20) . ' ...</p>',
    ];
  }

  /**
   * Gets the URI without the 'internal:' or 'entity:' scheme.
   *
   * The following two forms of URIs are transformed:
   * - 'entity:' URIs: to entity autocomplete ("label (entity id)") strings;
   * - 'internal:' URIs: the scheme is stripped.
   *
   * This method is the inverse of ::getUserEnteredStringAsUri().
   *
   * @param string $uri
   *   The URI to get the displayable string for.
   *
   * @return string
   *   URI as displayable string.
   *
   * @see static::getUserEnteredStringAsUri()
   */
  public static function getUriAsDisplayableString($uri) {
    $scheme = parse_url($uri, PHP_URL_SCHEME);

    // By default, the displayable string is the URI.
    $displayable_string = $uri;

    // A different displayable string may be chosen in case of the 'internal:'
    // or 'entity:' built-in schemes.
    if ($scheme === 'internal') {
      $uri_reference = explode(':', $uri, 2)[1];

      // @todo '<front>' is valid input for BC reasons, may be removed by
      //   https://www.drupal.org/node/2421941
      $path = parse_url($uri, PHP_URL_PATH);
      if ($path === '/') {
        $uri_reference = '<front>' . substr($uri_reference, 1);
      }

      $displayable_string = $uri_reference;
    }
    elseif ($scheme === 'entity') {
      [$entity_type, $entity_id] = explode('/', substr($uri, 7), 2);
      // Show the 'entity:' URI as the entity autocomplete would.
      // @todo Support entity types other than 'node'. Will be fixed in
      //   https://www.drupal.org/node/2423093.
      if ($entity_type == 'node' && $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id)) {
        $displayable_string = EntityAutocomplete::getEntityLabels([$entity]);
      }
    }
    elseif ($scheme === 'route') {
      $displayable_string = ltrim($displayable_string, 'route:');
    }

    return $displayable_string;
  }

}
