<?php

namespace Drupal\ultimate_table_field\Element;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Url;

/**
 * Ultimate table trait.
 */
trait UltimateTableTrait {

  /**
   * Get table first item ever.
   */
  public static function getFirstItemEver($i, $j, $values) {
    $item = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['table-first-item-ever'],
      ],
    ];

    // @todo Potentially using any cell field type here not only text.
    $item['text'] = [
      '#type' => 'textfield',
      '#placeholder' => t('First table cell text...'),
      '#size' => 50,
      '#value' => $values['columns'][$j] ?? NULL,
    ];
    return $item;
  }

  /**
   * Get Controlled items (item that control entire row and entire column).
   */
  public static function getControlledItem($id, $first_parent, $i, $j, $values, $type = 'row') {
    $position_class = $j === 0 && $i !== 0 ? 'utf-row-remove-button' : 'utf-column-remove-button';
    $item = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['table-controlled-item', $position_class],
      ],
    ];
    $item['remove'] = self::buildRemoveButton($id, $first_parent, $type, $i, $j);
    return $item;
  }

  /**
   * Build cell summary element.
   */
  public static function buildCellSummaryElement($id, $first_parent, &$element, $i, $j, $items_data, $cell_id, $ut_cell_field_manager, $allowed_types) {
    $element['table'][$i][$j]['container'] = [
      '#type' => 'fieldset',
      '#title' => t('Summary'),
      '#attributes' => [
        'class' => [empty($items_data) ? 'js-hide' : '', 'summary-container'],
      ],
      '#prefix' => "<div id=\"$cell_id\">",
      '#suffix' => '</div>',
    ];
    $query = [
      'items_data' => Json::encode($items_data),
      'op' => 'update',
      'row_index' => $i,
      'column_index' => $j,
      'allowed_types' => Json::encode($allowed_types),
      'wrapper_id' => $id,
    ];
    $items_data = empty($items_data) ? [] : $items_data;
    foreach ($items_data as $key => $item_data) {
      $plugin_id = $item_data['type'] ?? '';
      if (!$ut_cell_field_manager->hasDefinition($plugin_id)) {
        continue;
      }
      $definition = $ut_cell_field_manager->createInstance($plugin_id);
      $element['table'][$i][$j]['container'][$key] = [
        '#type' => 'fieldset',
        '#prefix' => "<div id=\"table-cell-container-$i-$j\">",
        '#suffix' => "</div>",
        '#attributes' => [
          'class' => ['table-cell-item-summary'],
        ],
      ];
      $element['table'][$i][$j]['container'][$key]['summary'] = $definition->generateSummary($item_data);
      $link_title = [
        '#markup' => '<span class="update-item-button"></span>',
      ];
      $query['item_index'] = $key;
      $element['table'][$i][$j]['container'][$key]['update'] = static::modalLinkBuilder($link_title, $query);
      $element['table'][$i][$j]['container'][$key]['remove'] = self::buildRemoveButton($id, $first_parent, 'item', $i, $j, $key);
    }
    // Modal link opener.
    $query['op'] = 'add';
    $element['table'][$i][$j]['open_modal'] = static::modalLinkBuilder(t('Add'), $query);
  }

  /**
   * Modal link builder.
   */
  protected static function modalLinkBuilder($title, $query) {
    $op = $query['op'] ?? '';
    return [
      '#type' => 'link',
      '#title' => $title,
      '#url' => Url::fromRoute('ultimate_table_field.open_modal_form', [], [
        'query' => $query,
      ]),
      '#attributes' => [
        'class' => [
          'use-ajax',
          $op === 'add' ? 'button' : '',
        ],
      ],
      '#prefix' => '<div class="table-cell-item-actions">',
    ];
  }

  /**
   * Build remove button element.
   */
  public static function buildRemoveButton($id, $first_parent, $type, $i, $j, $k = 0) {
    $name = $type . "_remove:";
    if ($type === 'row') {
      $name .= $i;
      $title = t('Remove row');
    }
    if ($type === 'column') {
      $name .= $j;
      $title = t('Remove column');
    }
    if ($type === 'item') {
      $name .= $i . '_' . $j . '_' . $k;
      $title = t('Remove item');
    }
    $message = $type == 'row' ? t('Deleting row...') : NULL;
    $message = !$message && $type == 'column' ? t('Deleting column...') : $message;
    $message = !$message && $type == 'item' ? t('Deleting item...') : $message;
    return [
      '#type' => 'submit',
      '#value' => ' ',
      '#submit' => $type !== 'item' ? [[self::class, 'elementSubmitCallback']] : [],
      '#name' => $name . '::' . str_replace('-', '_', $id),
      '#limit_validation_errors' => [[$first_parent]],
      '#attributes' => [
        'class' => ['button', 'button--danger', 'table-remove-img', 'use-ajax'],
        'title' => $title,
      ],
      '#suffix' => $type === 'item' ? '</div>' : '',
      '#ajax' => [
        'callback' => [self::class, 'updateTable'],
        'wrapper' => $id,
        'progress' => [
          'type' => 'throbber',
          'message' => $message,
        ],
      ],
    ];
  }

}
