<?php

namespace Drupal\ultimate_table_field\Plugin\UltimateTable\CellField;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\ultimate_table_field\UltimateTableCellFieldBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides file field cell form element.
 *
 * @UltimateTableCellField(
 *   id="file",
 *   label=@Translation("Default file"),
 * )
 */
class File extends UltimateTableCellFieldBase implements ContainerFactoryPluginInterface {

  /**
   * File system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * File system service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritDoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, FileSystemInterface $fileSystem, EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fileSystem = $fileSystem;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_system'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function buildCellField($default_value = NULL): array {
    $destination = 'public://ultimate-table/documents';
    $this->fileSystem->prepareDirectory($destination, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    return [
      '#type' => 'fieldset',
      '#tree' => TRUE,
      'file' => [
        '#type' => 'managed_file',
        '#title' => $this->t('Upload a document'),
        '#default_value' => $default_value['file'] ?? NULL,
        '#upload_validators' => [
          'FileExtension' => [
            'extensions' => 'pdf doc docx',
          ],
        ],
        '#upload_location' => 'public://ultimate-table/documents',
        '#description' => $this->t('Allowed file types: .pdf, .doc, .docx'),
      ],
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function cellFieldFormatter(array $item): array {
    $fid = $item[$this->id()]['file'][0] ?? NULL;
    $item = [
      '#type' => 'container',
      'link' => [
        '#markup' => $this->t('File not found'),
      ],
    ];
    if ($fid) {
      $file = $this->entityTypeManager->getStorage('file')->load($fid);
      if ($file) {
        $url = $file->createFileUrl();
        $item['link']['#markup'] = "<a href=\"$url\" download>" . $this->t('Download @filename', ['@filename' => $file->getFilename()]) . "</a>";
      }
    }
    return $item;
  }

  /**
   * {@inheritDoc}
   */
  public function generateSummary($data = NULL): array {
    $fid = $data[$this->id()]['file'][0] ?? NULL;
    $filename = '';
    if ($fid) {
      $file = $this->entityTypeManager->getStorage('file')->load($fid);
      if ($file) {
        $filename = $file->getFilename();
      }
    }
    $filename = !empty($filename) ? $this->t('<strong>File:</strong> @filename', ['@filename' => $filename]) : '';
    return [
      '#type' => 'markup',
      '#markup' => $this->t('<p><strong>Type:</strong> @label @filename', [
        '@label' => $this->label(),
        '@filename' => $filename,
      ]),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function cellFieldAlterSubmitted(mixed &$submitted_value) {
    $fid = $submitted_value['file'][0] ?? NULL;
    if (!$fid) {
      return;
    }
    $file = $this->entityTypeManager->getStorage('file')->load($fid);
    if (!$file) {
      return;
    }
    if (!$file->isPermanent()) {
      // The file as permanent.
      $file->setPermanent();
      $file->save();
    }
  }

}
