<?php

namespace Drupal\ultimate_table_field\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\TypedData\TraversableTypedDataInterface;

/**
 * Plugin implementation of the ultimate table field type.
 *
 * @FieldType(
 *   id = "ultimate_table",
 *   label = @Translation("Ultimate table"),
 *   description = @Translation("A field that stores table data."),
 *   default_widget = "ultimate_table",
 *   default_formatter = "ultimate_table",
 * )
 */
class UltimateTableFieldType extends FieldItemBase {

  /**
   * Ultimate table cell field manager plugin.
   *
   * @var \Drupal\ultimate_table_field\UltimateTableCellFieldManager
   */
  protected $ultimateTableCellFieldManager;

  /**
   * Current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * {@inheritDoc}
   */
  public static function createInstance($definition, $name = NULL, ?TraversableTypedDataInterface $parent = NULL) {
    $container = \Drupal::getContainer();
    $instance = parent::createInstance($definition, $name, $parent);
    $instance->ultimateTableCellFieldManager = $container->get('plugin.manager.ultimate_table_cell_field');
    $instance->request = $container->get('request_stack')->getCurrentRequest();
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = [];

    $properties['value'] = DataDefinition::create('any')
      ->setLabel(t('Table value'))
      ->setRequired(TRUE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'value' => [
          'type' => 'blob',
          'not null' => TRUE,
          'serialize' => TRUE,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    if ($this->request->isXmlHttpRequest()) {
      // Bypass validation check during table manipulation.
      return FALSE;
    }
    $columns = $this->value['columns'] ?? [];
    $rows = $this->value['rows'] ?? [];
    $columns = array_filter($columns, fn($el) => !empty($el));
    $rows = array_filter($rows, fn($el) => !empty(array_filter($el)));
    return empty($rows) && empty($columns);
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'allowed_types' => [],
      'enable_legend' => FALSE,
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritDoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::fieldSettingsForm($form, $form_state);
    $definitions = $this->ultimateTableCellFieldManager->getDefinitions();
    $options = array_map(fn($definition) => $definition['label'], $definitions);
    $form['allowed_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Cell allowed field types'),
      '#options' => $options,
      '#default_value' => $this->getSetting('allowed_types') ?? [],
      '#description' => $this->t('Select table cell allowed field types plugins, leave empty to allow all types'),
    ];

    $form['enable_legend'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable legend field'),
      '#default_value' => $this->getSetting('enable_legend'),
      '#description' => $this->t('Add a legend field to the table.'),
    ];

    return $form;
  }

}
